#include "td_main.h"
#include <raymath.h>
#include <rlgl.h>
#include <stdlib.h>
#include <math.h>

//# Variables
GUIState guiState = {0};
GameTime gameTime = {
  .fixedDeltaTime = 1.0f / 60.0f,
};

Model floorTileAModel = {0};
Model floorTileBModel = {0};
Model treeModel[2] = {0};
Model firTreeModel[2] = {0};
Model rockModels[5] = {0};
Model grassPatchModel[1] = {0};

Model pathArrowModel = {0};
Model greenArrowModel = {0};

Texture2D palette, spriteSheet;

Level levels[] = {
  [0] = {
    .state = LEVEL_STATE_BUILDING,
    .initialGold = 20,
    .waves[0] = {
      .enemyType = ENEMY_TYPE_MINION,
      .wave = 0,
      .count = 5,
      .interval = 2.5f,
      .delay = 1.0f,
      .spawnPosition = {2, 6},
    },
    .waves[1] = {
      .enemyType = ENEMY_TYPE_MINION,
      .wave = 0,
      .count = 5,
      .interval = 2.5f,
      .delay = 1.0f,
      .spawnPosition = {-2, 6},
    },
    .waves[2] = {
      .enemyType = ENEMY_TYPE_MINION,
      .wave = 1,
      .count = 20,
      .interval = 1.5f,
      .delay = 1.0f,
      .spawnPosition = {0, 6},
    },
    .waves[3] = {
      .enemyType = ENEMY_TYPE_MINION,
      .wave = 2,
      .count = 30,
      .interval = 1.2f,
      .delay = 1.0f,
      .spawnPosition = {0, 6},
    }
  },
};

Level *currentLevel = levels;

//# Game

static Model LoadGLBModel(char *filename)
{
  Model model = LoadModel(TextFormat("data/%s.glb",filename));
  for (int i = 0; i < model.materialCount; i++)
  {
    model.materials[i].maps[MATERIAL_MAP_DIFFUSE].texture = palette;
  }
  return model;
}

void LoadAssets()
{
  // load a sprite sheet that contains all units
  spriteSheet = LoadTexture("data/spritesheet.png");
  SetTextureFilter(spriteSheet, TEXTURE_FILTER_BILINEAR);

  // we'll use a palette texture to colorize the all buildings and environment art
  palette = LoadTexture("data/palette.png");
  // The texture uses gradients on very small space, so we'll enable bilinear filtering
  SetTextureFilter(palette, TEXTURE_FILTER_BILINEAR);

  floorTileAModel = LoadGLBModel("floor-tile-a");
  floorTileBModel = LoadGLBModel("floor-tile-b");
  treeModel[0] = LoadGLBModel("leaftree-large-1-a");
  treeModel[1] = LoadGLBModel("leaftree-large-1-b");
  firTreeModel[0] = LoadGLBModel("firtree-1-a");
  firTreeModel[1] = LoadGLBModel("firtree-1-b");
  rockModels[0] = LoadGLBModel("rock-1");
  rockModels[1] = LoadGLBModel("rock-2");
  rockModels[2] = LoadGLBModel("rock-3");
  rockModels[3] = LoadGLBModel("rock-4");
  rockModels[4] = LoadGLBModel("rock-5");
  grassPatchModel[0] = LoadGLBModel("grass-patch-1");

  pathArrowModel = LoadGLBModel("direction-arrow-x");
  greenArrowModel = LoadGLBModel("green-arrow");
}

void InitLevel(Level *level)
{
  level->seed = (int)(GetTime() * 100.0f);

  TowerInit();
  EnemyInit();
  ProjectileInit();
  ParticleInit();
  TowerTryAdd(TOWER_TYPE_BASE, 0, 0);

  level->placementMode = 0;
  level->state = LEVEL_STATE_BUILDING;
  level->nextState = LEVEL_STATE_NONE;
  level->playerGold = level->initialGold;
  level->currentWave = 0;
  level->placementX = -1;
  level->placementY = 0;

  Camera *camera = &level->camera;
  camera->position = (Vector3){4.0f, 8.0f, 8.0f};
  camera->target = (Vector3){0.0f, 0.0f, 0.0f};
  camera->up = (Vector3){0.0f, 1.0f, 0.0f};
  camera->fovy = 10.0f;
  camera->projection = CAMERA_ORTHOGRAPHIC;
}

void DrawLevelHud(Level *level)
{
  const char *text = TextFormat("Gold: %d", level->playerGold);
  Font font = GetFontDefault();
  DrawTextEx(font, text, (Vector2){GetScreenWidth() - 120, 10}, font.baseSize * 2.0f, 2.0f, BLACK);
  DrawTextEx(font, text, (Vector2){GetScreenWidth() - 122, 8}, font.baseSize * 2.0f, 2.0f, YELLOW);
}

void DrawLevelReportLostWave(Level *level)
{
  BeginMode3D(level->camera);
  DrawLevelGround(level);
  TowerDraw();
  EnemyDraw();
  ProjectileDraw();
  ParticleDraw();
  guiState.isBlocked = 0;
  EndMode3D();

  TowerDrawHealthBars(level->camera);

  const char *text = "Wave lost";
  int textWidth = MeasureText(text, 20);
  DrawText(text, (GetScreenWidth() - textWidth) * 0.5f, 20, 20, WHITE);

  if (Button("Reset level", 20, GetScreenHeight() - 40, 160, 30, 0))
  {
    level->nextState = LEVEL_STATE_RESET;
  }
}

int HasLevelNextWave(Level *level)
{
  for (int i = 0; i < 10; i++)
  {
    EnemyWave *wave = &level->waves[i];
    if (wave->wave == level->currentWave)
    {
      return 1;
    }
  }
  return 0;
}

void DrawLevelReportWonWave(Level *level)
{
  BeginMode3D(level->camera);
  DrawLevelGround(level);
  TowerDraw();
  EnemyDraw();
  ProjectileDraw();
  ParticleDraw();
  guiState.isBlocked = 0;
  EndMode3D();

  TowerDrawHealthBars(level->camera);

  const char *text = "Wave won";
  int textWidth = MeasureText(text, 20);
  DrawText(text, (GetScreenWidth() - textWidth) * 0.5f, 20, 20, WHITE);


  if (Button("Reset level", 20, GetScreenHeight() - 40, 160, 30, 0))
  {
    level->nextState = LEVEL_STATE_RESET;
  }

  if (HasLevelNextWave(level))
  {
    if (Button("Prepare for next wave", GetScreenWidth() - 300, GetScreenHeight() - 40, 300, 30, 0))
    {
      level->nextState = LEVEL_STATE_BUILDING;
    }
  }
  else {
    if (Button("Level won", GetScreenWidth() - 300, GetScreenHeight() - 40, 300, 30, 0))
    {
      level->nextState = LEVEL_STATE_WON_LEVEL;
    }
  }
}

void DrawBuildingBuildButton(Level *level, int x, int y, int width, int height, uint8_t towerType, const char *name)
{
  static ButtonState buttonStates[8] = {0};
  int cost = GetTowerCosts(towerType);
  const char *text = TextFormat("%s: %d", name, cost);
  buttonStates[towerType].isSelected = level->placementMode == towerType;
  buttonStates[towerType].isDisabled = level->playerGold < cost;
  if (Button(text, x, y, width, height, &buttonStates[towerType]))
  {
    level->placementMode = buttonStates[towerType].isSelected ? 0 : towerType;
    level->nextState = LEVEL_STATE_BUILDING_PLACEMENT;
  }
}

float GetRandomFloat(float min, float max)
{
  int random = GetRandomValue(0, 0xfffffff);
  return ((float)random / (float)0xfffffff) * (max - min) + min;
}

void DrawLevelGround(Level *level)
{
  // draw checkerboard ground pattern
  for (int x = -5; x <= 5; x += 1)
  {
    for (int y = -5; y <= 5; y += 1)
    {
      Model *model = (x + y) % 2 == 0 ? &floorTileAModel : &floorTileBModel;
      DrawModel(*model, (Vector3){x, 0.0f, y}, 1.0f, WHITE);
    }
  }

  int oldSeed = GetRandomValue(0, 0xfffffff);
  SetRandomSeed(level->seed);
  // increase probability for trees via duplicated entries
  Model borderModels[64];
  int maxRockCount = GetRandomValue(2, 6);
  int maxTreeCount = GetRandomValue(10, 20);
  int maxFirTreeCount = GetRandomValue(5, 10);
  int maxLeafTreeCount = maxTreeCount - maxFirTreeCount;
  int grassPatchCount = GetRandomValue(5, 30);

  int modelCount = 0;
  for (int i = 0; i < maxRockCount && modelCount < 63; i++)
  {
    borderModels[modelCount++] = rockModels[GetRandomValue(0, 5)];
  }
  for (int i = 0; i < maxLeafTreeCount && modelCount < 63; i++)
  {
    borderModels[modelCount++] = treeModel[GetRandomValue(0, 1)];
  }
  for (int i = 0; i < maxFirTreeCount && modelCount < 63; i++)
  {
    borderModels[modelCount++] = firTreeModel[GetRandomValue(0, 1)];
  }
  for (int i = 0; i < grassPatchCount && modelCount < 63; i++)
  {
    borderModels[modelCount++] = grassPatchModel[0];
  }

  // draw some objects around the border of the map
  Vector3 up = {0, 1, 0};
  // a pseudo random number generator to get the same result every time
  const float wiggle = 0.75f;
  const int layerCount = 3;
  for (int layer = 0; layer < layerCount; layer++)
  {
    int layerPos = 6 + layer;
    for (int x = -6 + layer; x <= 6 + layer; x += 1)
    {
      DrawModelEx(borderModels[GetRandomValue(0, modelCount - 1)], 
        (Vector3){x + GetRandomFloat(0.0f, wiggle), 0.0f, -layerPos + GetRandomFloat(0.0f, wiggle)}, 
        up, GetRandomFloat(0.0f, 360), Vector3One(), WHITE);
      DrawModelEx(borderModels[GetRandomValue(0, modelCount - 1)], 
        (Vector3){x + GetRandomFloat(0.0f, wiggle), 0.0f, layerPos + GetRandomFloat(0.0f, wiggle)}, 
        up, GetRandomFloat(0.0f, 360), Vector3One(), WHITE);
    }

    for (int z = -5 + layer; z <= 5 + layer; z += 1)
    {
      DrawModelEx(borderModels[GetRandomValue(0, modelCount - 1)], 
        (Vector3){-layerPos + GetRandomFloat(0.0f, wiggle), 0.0f, z + GetRandomFloat(0.0f, wiggle)}, 
        up, GetRandomFloat(0.0f, 360), Vector3One(), WHITE);
      DrawModelEx(borderModels[GetRandomValue(0, modelCount - 1)], 
        (Vector3){layerPos + GetRandomFloat(0.0f, wiggle), 0.0f, z + GetRandomFloat(0.0f, wiggle)}, 
        up, GetRandomFloat(0.0f, 360), Vector3One(), WHITE);
    }
  }

  SetRandomSeed(oldSeed);
}

void DrawEnemyPath(Level *level, Color arrowColor)
{
  const int castleX = 0, castleY = 0;
  const int maxWaypointCount = 200;
  const float timeStep = 1.0f;
  Vector3 arrowScale = {0.75f, 0.75f, 0.75f};

  // we start with a time offset to simulate the path, 
  // this way the arrows are animated in a forward moving direction
  // The time is wrapped around the time step to get a smooth animation
  float timeOffset = fmodf(GetTime(), timeStep);

  for (int i = 0; i < ENEMY_MAX_WAVE_COUNT; i++)
  {
    EnemyWave *wave = &level->waves[i];
    if (wave->wave != level->currentWave)
    {
      continue;
    }

    // use this dummy enemy to simulate the path
    Enemy dummy = {
      .enemyType = ENEMY_TYPE_MINION,
      .simPosition = (Vector2){wave->spawnPosition.x, wave->spawnPosition.y},
      .nextX = wave->spawnPosition.x,
      .nextY = wave->spawnPosition.y,
      .currentX = wave->spawnPosition.x,
      .currentY = wave->spawnPosition.y,
    };

    float deltaTime = timeOffset;
    for (int j = 0; j < maxWaypointCount; j++)
    {
      int waypointPassedCount = 0;
      Vector2 pos = EnemyGetPosition(&dummy, deltaTime, &dummy.simVelocity, &waypointPassedCount);
      // after the initial variable starting offset, we use a fixed time step
      deltaTime = timeStep;
      dummy.simPosition = pos;

      // Update the dummy's position just like we do in the regular enemy update loop
      for (int k = 0; k < waypointPassedCount; k++)
      {
        dummy.currentX = dummy.nextX;
        dummy.currentY = dummy.nextY;
        if (EnemyGetNextPosition(dummy.currentX, dummy.currentY, &dummy.nextX, &dummy.nextY) &&
          Vector2DistanceSqr(dummy.simPosition, (Vector2){castleX, castleY}) <= 0.25f * 0.25f)
        {
          break;
        }
      }
      if (Vector2DistanceSqr(dummy.simPosition, (Vector2){castleX, castleY}) <= 0.25f * 0.25f)
      {
        break;
      }
      
      // get the angle we need to rotate the arrow model. The velocity is just fine for this.
      float angle = atan2f(dummy.simVelocity.x, dummy.simVelocity.y) * RAD2DEG - 90.0f;
      DrawModelEx(pathArrowModel, (Vector3){pos.x, 0.15f, pos.y}, (Vector3){0, 1, 0}, angle, arrowScale, arrowColor);
    }
  }
}

void DrawEnemyPaths(Level *level)
{
  // disable depth testing for the path arrows
  // flush the 3D batch to draw the arrows on top of everything
  rlDrawRenderBatchActive();
  rlDisableDepthTest();
  DrawEnemyPath(level, (Color){64, 64, 64, 160});

  rlDrawRenderBatchActive();
  rlEnableDepthTest();
  DrawEnemyPath(level, WHITE);
}

static void SimulateTowerPlacementBehavior(Level *level, float towerFloatHeight, float mapX, float mapY)
{
  float dt = gameTime.fixedDeltaTime;
  // smooth transition for the placement position using exponential decay
  const float lambda = 15.0f;
  float factor = 1.0f - expf(-lambda * dt);

  float damping = 0.5f;
  float springStiffness = 300.0f;
  float springDecay = 95.0f;
  float minHeight = 0.35f;

  if (level->placementPhase == PLACEMENT_PHASE_STARTING)
  {
    damping = 1.0f;
    springDecay = 90.0f;
    springStiffness = 100.0f;
    minHeight = 0.70f;
  }

  for (int i = 0; i < gameTime.fixedStepCount; i++)
  {
    level->placementTransitionPosition = 
      Vector2Lerp(
        level->placementTransitionPosition, 
        (Vector2){mapX, mapY}, factor);

    // draw the spring position for debugging the spring simulation
    // first step: stiff spring, no simulation
    Vector3 worldPlacementPosition = (Vector3){
      level->placementTransitionPosition.x, towerFloatHeight, level->placementTransitionPosition.y};
    Vector3 springTargetPosition = (Vector3){
      worldPlacementPosition.x, 1.0f + towerFloatHeight, worldPlacementPosition.z};
    // consider the current velocity to predict the future position in order to dampen
    // the spring simulation. Longer prediction times will result in more damping
    Vector3 predictedSpringPosition = Vector3Add(level->placementTowerSpring.position, 
      Vector3Scale(level->placementTowerSpring.velocity, dt * damping));
    Vector3 springPointDelta = Vector3Subtract(springTargetPosition, predictedSpringPosition);
    Vector3 velocityChange = Vector3Scale(springPointDelta, dt * springStiffness);
    // decay velocity of the upright forcing spring
    // This force acts like a 2nd spring that pulls the tip upright into the air above the
    // base position
    level->placementTowerSpring.velocity = Vector3Scale(level->placementTowerSpring.velocity, 1.0f - expf(-springDecay * dt));
    level->placementTowerSpring.velocity = Vector3Add(level->placementTowerSpring.velocity, velocityChange);

    // calculate length of the spring to calculate the force to apply to the placementTowerSpring position
    // we use a simple spring model with a rest length of 1.0f
    Vector3 springDelta = Vector3Subtract(worldPlacementPosition, level->placementTowerSpring.position);
    float springLength = Vector3Length(springDelta);
    float springForce = (springLength - 1.0f) * springStiffness;
    Vector3 springForceVector = Vector3Normalize(springDelta);
    springForceVector = Vector3Scale(springForceVector, springForce);
    level->placementTowerSpring.velocity = Vector3Add(level->placementTowerSpring.velocity, 
      Vector3Scale(springForceVector, dt));

    level->placementTowerSpring.position = Vector3Add(level->placementTowerSpring.position, 
      Vector3Scale(level->placementTowerSpring.velocity, dt));
    if (level->placementTowerSpring.position.y < minHeight + towerFloatHeight)
    {
      level->placementTowerSpring.velocity.y *= -1.0f;
      level->placementTowerSpring.position.y = fmaxf(level->placementTowerSpring.position.y, minHeight + towerFloatHeight);
    }
  }
}

void DrawLevelBuildingPlacementState(Level *level)
{
  const float placementDuration = 0.5f;

  level->placementTimer += gameTime.deltaTime;
  if (level->placementTimer > 1.0f && level->placementPhase == PLACEMENT_PHASE_STARTING)
  {
    level->placementPhase = PLACEMENT_PHASE_MOVING;
    level->placementTimer = 0.0f;
  }

  BeginMode3D(level->camera);
  DrawLevelGround(level);

  int blockedCellCount = 0;
  Vector2 blockedCells[1];
  Ray ray = GetScreenToWorldRay(GetMousePosition(), level->camera);
  float planeDistance = ray.position.y / -ray.direction.y;
  float planeX = ray.direction.x * planeDistance + ray.position.x;
  float planeY = ray.direction.z * planeDistance + ray.position.z;
  int16_t mapX = (int16_t)floorf(planeX + 0.5f);
  int16_t mapY = (int16_t)floorf(planeY + 0.5f);
  if (level->placementPhase == PLACEMENT_PHASE_MOVING && 
    level->placementMode && !guiState.isBlocked && 
    mapX >= -5 && mapX <= 5 && mapY >= -5 && mapY <= 5 && IsMouseButtonDown(MOUSE_LEFT_BUTTON))
  {
    level->placementX = mapX;
    level->placementY = mapY;
  }
  else
  {
    mapX = level->placementX;
    mapY = level->placementY;
  }
  blockedCells[blockedCellCount++] = (Vector2){mapX, mapY};
  PathFindingMapUpdate(blockedCellCount, blockedCells);

  TowerDraw();
  EnemyDraw();
  ProjectileDraw();
  ParticleDraw();
  DrawEnemyPaths(level);

  // let the tower float up and down. Consider this height in the spring simulation as well
  float towerFloatHeight = sinf(gameTime.time * 4.0f) * 0.2f + 0.3f;

  if (level->placementPhase == PLACEMENT_PHASE_PLACING)
  {
    // The bouncing spring needs a bit of outro time to look nice and complete. 
    // So we scale the time so that the first 2/3rd of the placing phase handles the motion
    // and the last 1/3rd is the outro physics (bouncing)
    float t = fminf(1.0f, level->placementTimer / placementDuration * 1.5f);
    // let towerFloatHeight describe parabola curve, starting at towerFloatHeight and ending at 0
    float linearBlendHeight = (1.0f - t) * towerFloatHeight;
    float parabola = (1.0f - ((t - 0.5f) * (t - 0.5f) * 4.0f)) * 2.0f;
    towerFloatHeight = linearBlendHeight + parabola;
  }

  SimulateTowerPlacementBehavior(level, towerFloatHeight, mapX, mapY);
  
  rlPushMatrix();
  rlTranslatef(level->placementTransitionPosition.x, 0, level->placementTransitionPosition.y);

  rlPushMatrix();
  rlTranslatef(0.0f, towerFloatHeight, 0.0f);
  // calculate x and z rotation to align the model with the spring
  Vector3 position = {level->placementTransitionPosition.x, towerFloatHeight, level->placementTransitionPosition.y};
  Vector3 towerUp = Vector3Subtract(level->placementTowerSpring.position, position);
  Vector3 rotationAxis = Vector3CrossProduct(towerUp, (Vector3){0, 1, 0});
  float angle = acosf(Vector3DotProduct(towerUp, (Vector3){0, 1, 0}) / Vector3Length(towerUp)) * RAD2DEG;
  float springLength = Vector3Length(towerUp);
  float towerStretch = fminf(fmaxf(springLength, 0.5f), 4.5f);
  float towerSquash = 1.0f / towerStretch;
  rlRotatef(-angle, rotationAxis.x, rotationAxis.y, rotationAxis.z);
  rlScalef(towerSquash, towerStretch, towerSquash);
  Tower dummy = {
    .towerType = level->placementMode,
  };
  TowerDrawSingle(dummy);
  rlPopMatrix();

  // draw a shadow for the tower
  float umbrasize = 0.8 + sqrtf(towerFloatHeight);
  DrawCube((Vector3){0.0f, 0.05f, 0.0f}, umbrasize, 0.0f, umbrasize, (Color){0, 0, 0, 32});
  DrawCube((Vector3){0.0f, 0.075f, 0.0f}, 0.85f, 0.0f, 0.85f, (Color){0, 0, 0, 64});


  float bounce = sinf(gameTime.time * 8.0f) * 0.5f + 0.5f;
  float offset = fmaxf(0.0f, bounce - 0.4f) * 0.35f + 0.7f;
  float squeeze = -fminf(0.0f, bounce - 0.4f) * 0.7f + 0.7f;
  float stretch = fmaxf(0.0f, bounce - 0.3f) * 0.5f + 0.8f;
  
  DrawModelEx(greenArrowModel, (Vector3){ offset, 0.0f,  0.0f}, (Vector3){0, 1, 0}, 90, (Vector3){squeeze, 1.0f, stretch}, WHITE);
  DrawModelEx(greenArrowModel, (Vector3){-offset, 0.0f,  0.0f}, (Vector3){0, 1, 0}, 270, (Vector3){squeeze, 1.0f, stretch}, WHITE);
  DrawModelEx(greenArrowModel, (Vector3){ 0.0f, 0.0f,  offset}, (Vector3){0, 1, 0}, 0, (Vector3){squeeze, 1.0f, stretch}, WHITE);
  DrawModelEx(greenArrowModel, (Vector3){ 0.0f, 0.0f, -offset}, (Vector3){0, 1, 0}, 180, (Vector3){squeeze, 1.0f, stretch}, WHITE);
  rlPopMatrix();

  guiState.isBlocked = 0;

  EndMode3D();

  TowerDrawHealthBars(level->camera);

  if (level->placementPhase == PLACEMENT_PHASE_PLACING)
  {
    if (level->placementTimer > placementDuration)
    {
        TowerTryAdd(level->placementMode, mapX, mapY);
        level->playerGold -= GetTowerCosts(level->placementMode);
        level->nextState = LEVEL_STATE_BUILDING;
        level->placementMode = TOWER_TYPE_NONE;
    }
  }
  else
  {   
    if (Button("Cancel", 20, GetScreenHeight() - 40, 160, 30, 0))
    {
      level->nextState = LEVEL_STATE_BUILDING;
      level->placementMode = TOWER_TYPE_NONE;
      TraceLog(LOG_INFO, "Cancel building");
    }
    
    if (TowerGetAt(mapX, mapY) == 0 &&  Button("Build", GetScreenWidth() - 180, GetScreenHeight() - 40, 160, 30, 0))
    {
      level->placementPhase = PLACEMENT_PHASE_PLACING;
      level->placementTimer = 0.0f;
    }
  }
}

void DrawLevelBuildingState(Level *level)
{
  BeginMode3D(level->camera);
  DrawLevelGround(level);

  PathFindingMapUpdate(0, 0);
  TowerDraw();
  EnemyDraw();
  ProjectileDraw();
  ParticleDraw();
  DrawEnemyPaths(level);

  guiState.isBlocked = 0;

  EndMode3D();

  TowerDrawHealthBars(level->camera);

  DrawBuildingBuildButton(level, 10, 10, 110, 30, TOWER_TYPE_WALL, "Wall");
  DrawBuildingBuildButton(level, 10, 50, 110, 30, TOWER_TYPE_ARCHER, "Archer");
  DrawBuildingBuildButton(level, 10, 90, 110, 30, TOWER_TYPE_BALLISTA, "Ballista");
  DrawBuildingBuildButton(level, 10, 130, 110, 30, TOWER_TYPE_CATAPULT, "Catapult");

  if (Button("Reset level", 20, GetScreenHeight() - 40, 160, 30, 0))
  {
    level->nextState = LEVEL_STATE_RESET;
  }
  
  if (Button("Begin waves", GetScreenWidth() - 160, GetScreenHeight() - 40, 160, 30, 0))
  {
    level->nextState = LEVEL_STATE_BATTLE;
  }

  const char *text = "Building phase";
  int textWidth = MeasureText(text, 20);
  DrawText(text, (GetScreenWidth() - textWidth) * 0.5f, 20, 20, WHITE);
}

void InitBattleStateConditions(Level *level)
{
  level->state = LEVEL_STATE_BATTLE;
  level->nextState = LEVEL_STATE_NONE;
  level->waveEndTimer = 0.0f;
  for (int i = 0; i < 10; i++)
  {
    EnemyWave *wave = &level->waves[i];
    wave->spawned = 0;
    wave->timeToSpawnNext = wave->delay;
  }
}

void DrawLevelBattleState(Level *level)
{
  BeginMode3D(level->camera);
  DrawLevelGround(level);
  TowerDraw();
  EnemyDraw();
  ProjectileDraw();
  ParticleDraw();
  guiState.isBlocked = 0;
  EndMode3D();

  EnemyDrawHealthbars(level->camera);
  TowerDrawHealthBars(level->camera);

  if (Button("Reset level", 20, GetScreenHeight() - 40, 160, 30, 0))
  {
    level->nextState = LEVEL_STATE_RESET;
  }

  int maxCount = 0;
  int remainingCount = 0;
  for (int i = 0; i < 10; i++)
  {
    EnemyWave *wave = &level->waves[i];
    if (wave->wave != level->currentWave)
    {
      continue;
    }
    maxCount += wave->count;
    remainingCount += wave->count - wave->spawned;
  }
  int aliveCount = EnemyCount();
  remainingCount += aliveCount;

  const char *text = TextFormat("Battle phase: %03d%%", 100 - remainingCount * 100 / maxCount);
  int textWidth = MeasureText(text, 20);
  DrawText(text, (GetScreenWidth() - textWidth) * 0.5f, 20, 20, WHITE);
}

void DrawLevel(Level *level)
{
  switch (level->state)
  {
    case LEVEL_STATE_BUILDING: DrawLevelBuildingState(level); break;
    case LEVEL_STATE_BUILDING_PLACEMENT: DrawLevelBuildingPlacementState(level); break;
    case LEVEL_STATE_BATTLE: DrawLevelBattleState(level); break;
    case LEVEL_STATE_WON_WAVE: DrawLevelReportWonWave(level); break;
    case LEVEL_STATE_LOST_WAVE: DrawLevelReportLostWave(level); break;
    default: break;
  }

  DrawLevelHud(level);
}

void UpdateLevel(Level *level)
{
  if (level->state == LEVEL_STATE_BATTLE)
  {
    int activeWaves = 0;
    for (int i = 0; i < 10; i++)
    {
      EnemyWave *wave = &level->waves[i];
      if (wave->spawned >= wave->count || wave->wave != level->currentWave)
      {
        continue;
      }
      activeWaves++;
      wave->timeToSpawnNext -= gameTime.deltaTime;
      if (wave->timeToSpawnNext <= 0.0f)
      {
        Enemy *enemy = EnemyTryAdd(wave->enemyType, wave->spawnPosition.x, wave->spawnPosition.y);
        if (enemy)
        {
          wave->timeToSpawnNext = wave->interval;
          wave->spawned++;
        }
      }
    }
    if (GetTowerByType(TOWER_TYPE_BASE) == 0) {
      level->waveEndTimer += gameTime.deltaTime;
      if (level->waveEndTimer >= 2.0f)
      {
        level->nextState = LEVEL_STATE_LOST_WAVE;
      }
    }
    else if (activeWaves == 0 && EnemyCount() == 0)
    {
      level->waveEndTimer += gameTime.deltaTime;
      if (level->waveEndTimer >= 2.0f)
      {
        level->nextState = LEVEL_STATE_WON_WAVE;
      }
    }
  }

  PathFindingMapUpdate(0, 0);
  EnemyUpdate();
  TowerUpdate();
  ProjectileUpdate();
  ParticleUpdate();

  if (level->nextState == LEVEL_STATE_RESET)
  {
    InitLevel(level);
  }
  
  if (level->nextState == LEVEL_STATE_BATTLE)
  {
    InitBattleStateConditions(level);
  }
  
  if (level->nextState == LEVEL_STATE_WON_WAVE)
  {
    level->currentWave++;
    level->state = LEVEL_STATE_WON_WAVE;
  }
  
  if (level->nextState == LEVEL_STATE_LOST_WAVE)
  {
    level->state = LEVEL_STATE_LOST_WAVE;
  }

  if (level->nextState == LEVEL_STATE_BUILDING)
  {
    level->state = LEVEL_STATE_BUILDING;
  }

  if (level->nextState == LEVEL_STATE_BUILDING_PLACEMENT)
  {
    level->state = LEVEL_STATE_BUILDING_PLACEMENT;
    level->placementTransitionPosition = (Vector2){
      level->placementX, level->placementY};
    // initialize the spring to the current position
    level->placementTowerSpring = (PhysicsPoint){
      .position = (Vector3){level->placementX, 8.0f, level->placementY},
      .velocity = (Vector3){0.0f, 0.0f, 0.0f},
    };
    level->placementPhase = PLACEMENT_PHASE_STARTING;
    level->placementTimer = 0.0f;
  }

  if (level->nextState == LEVEL_STATE_WON_LEVEL)
  {
    // make something of this later
    InitLevel(level);
  }

  level->nextState = LEVEL_STATE_NONE;
}

float nextSpawnTime = 0.0f;

void ResetGame()
{
  InitLevel(currentLevel);
}

void InitGame()
{
  TowerInit();
  EnemyInit();
  ProjectileInit();
  ParticleInit();
  PathfindingMapInit(20, 20, (Vector3){-10.0f, 0.0f, -10.0f}, 1.0f);

  currentLevel = levels;
  InitLevel(currentLevel);
}

//# Immediate GUI functions

void DrawHealthBar(Camera3D camera, Vector3 position, float healthRatio, Color barColor, float healthBarWidth)
{
  const float healthBarHeight = 6.0f;
  const float healthBarOffset = 15.0f;
  const float inset = 2.0f;
  const float innerWidth = healthBarWidth - inset * 2;
  const float innerHeight = healthBarHeight - inset * 2;

  Vector2 screenPos = GetWorldToScreen(position, camera);
  float centerX = screenPos.x - healthBarWidth * 0.5f;
  float topY = screenPos.y - healthBarOffset;
  DrawRectangle(centerX, topY, healthBarWidth, healthBarHeight, BLACK);
  float healthWidth = innerWidth * healthRatio;
  DrawRectangle(centerX + inset, topY + inset, healthWidth, innerHeight, barColor);
}

int Button(const char *text, int x, int y, int width, int height, ButtonState *state)
{
  Rectangle bounds = {x, y, width, height};
  int isPressed = 0;
  int isSelected = state && state->isSelected;
  int isDisabled = state && state->isDisabled;
  if (CheckCollisionPointRec(GetMousePosition(), bounds) && !guiState.isBlocked && !isDisabled)
  {
    Color color = isSelected ? DARKGRAY : GRAY;
    DrawRectangle(x, y, width, height, color);
    if (IsMouseButtonPressed(MOUSE_LEFT_BUTTON))
    {
      isPressed = 1;
    }
    guiState.isBlocked = 1;
  }
  else
  {
    Color color = isSelected ? WHITE : LIGHTGRAY;
    DrawRectangle(x, y, width, height, color);
  }
  Font font = GetFontDefault();
  Vector2 textSize = MeasureTextEx(font, text, font.baseSize * 2.0f, 1);
  Color textColor = isDisabled ? GRAY : BLACK;
  DrawTextEx(font, text, (Vector2){x + width / 2 - textSize.x / 2, y + height / 2 - textSize.y / 2}, font.baseSize * 2.0f, 1, textColor);
  return isPressed;
}

//# Main game loop

void GameUpdate()
{
  UpdateLevel(currentLevel);
}

int main(void)
{
  int screenWidth, screenHeight;
  GetPreferredSize(&screenWidth, &screenHeight);
  InitWindow(screenWidth, screenHeight, "Tower defense");
  float gamespeed = 1.0f;
  SetTargetFPS(30);

  LoadAssets();
  InitGame();

  float pause = 1.0f;

  while (!WindowShouldClose())
  {
    if (IsPaused()) {
      // canvas is not visible in browser - do nothing
      continue;
    }

    if (IsKeyPressed(KEY_T))
    {
      gamespeed += 0.1f;
      if (gamespeed > 1.05f) gamespeed = 0.1f;
    }

    if (IsKeyPressed(KEY_P))
    {
      pause = pause > 0.5f ? 0.0f : 1.0f;
    }

    float dt = GetFrameTime() * gamespeed * pause;
    // cap maximum delta time to 0.1 seconds to prevent large time steps
    if (dt > 0.1f) dt = 0.1f;
    gameTime.time += dt;
    gameTime.deltaTime = dt;
    gameTime.frameCount += 1;

    float fixedTimeDiff = gameTime.time - gameTime.fixedTimeStart;
    gameTime.fixedStepCount = (uint8_t)(fixedTimeDiff / gameTime.fixedDeltaTime);

    BeginDrawing();
    ClearBackground((Color){0x4E, 0x63, 0x26, 0xFF});

    GameUpdate();
    DrawLevel(currentLevel);

    DrawText(TextFormat("Speed: %.1f", gamespeed), GetScreenWidth() - 180, 60, 20, WHITE);
    EndDrawing();

    gameTime.fixedTimeStart += gameTime.fixedStepCount * gameTime.fixedDeltaTime;
  }

  CloseWindow();

  return 0;
}