#include "td_main.h"
#include <raymath.h>

static TowerTypeConfig towerTypeConfigs[TOWER_TYPE_COUNT] = {
    [TOWER_TYPE_BASE] = {
        .name = "Castle",
        .maxHealth = 10,
    },
    [TOWER_TYPE_ARCHER] = {
        .name = "Archer",
        .cooldown = 0.5f,
        .maxUpgradeCooldown = 0.25f,
        .range = 3.0f,
        .maxUpgradeRange = 5.0f,
        .cost = 6,
        .maxHealth = 10,
        .projectileSpeed = 4.0f,
        .projectileType = PROJECTILE_TYPE_ARROW,
        .hitEffect = {
          .damage = 3.0f,
          .maxUpgradeDamage = 6.0f,
        },
    },
    [TOWER_TYPE_BALLISTA] = {
        .name = "Ballista",
        .cooldown = 1.5f,
        .maxUpgradeCooldown = 1.0f,
        .range = 6.0f,
        .maxUpgradeRange = 8.0f,
        .cost = 9,
        .maxHealth = 10,
        .projectileSpeed = 10.0f,
        .projectileType = PROJECTILE_TYPE_BALLISTA,
        .hitEffect = {
          .damage = 8.0f,
          .maxUpgradeDamage = 16.0f,
          .pushbackPowerDistance = 0.25f,
        }
    },
    [TOWER_TYPE_CATAPULT] = {
        .name = "Catapult",
        .cooldown = 1.7f,
        .maxUpgradeCooldown = 1.0f,
        .range = 5.0f,
        .maxUpgradeRange = 7.0f,
        .cost = 10,
        .maxHealth = 10,
        .projectileSpeed = 3.0f,
        .projectileType = PROJECTILE_TYPE_CATAPULT,
        .hitEffect = {
          .damage = 2.0f,
          .maxUpgradeDamage = 4.0f,
          .areaDamageRadius = 1.75f,
        }
    },
    [TOWER_TYPE_WALL] = {
        .name = "Wall",
        .cost = 2,
        .maxHealth = 10,
    },
};

Tower towers[TOWER_MAX_COUNT];
int towerCount = 0;

Model towerModels[TOWER_TYPE_COUNT];

// definition of our archer unit
SpriteUnit archerUnit = {
  .animations[0] = {
    .srcRect = {0, 0, 16, 16},
    .offset = {7, 1},
    .frameCount = 1,
    .frameDuration = 0.0f,
  },
  .animations[1] = {
    .animationId = SPRITE_UNIT_PHASE_WEAPON_COOLDOWN,
    .srcRect = {16, 0, 6, 16},
    .offset = {8, 0},
  },
  .animations[2] = {
    .animationId = SPRITE_UNIT_PHASE_WEAPON_IDLE,
    .srcRect = {22, 0, 11, 16},
    .offset = {10, 0},
  },
};

void DrawSpriteUnit(SpriteUnit unit, Vector3 position, float t, int flip, int phase)
{
  float unitScale = unit.scale == 0 ? 1.0f : unit.scale;
  float xScale = flip ? -1.0f : 1.0f;
  Camera3D camera = currentLevel->camera;
  float size = 0.5f * unitScale;
  // we want the sprite to face the camera, so we need to calculate the up vector
  Vector3 forward = Vector3Subtract(camera.target, camera.position);
  Vector3 up = {0, 1, 0};
  Vector3 right = Vector3CrossProduct(forward, up);
  up = Vector3Normalize(Vector3CrossProduct(right, forward));
  
  for (int i=0;i<SPRITE_UNIT_ANIMATION_COUNT;i++)
  {
    SpriteAnimation anim = unit.animations[i];
    if (anim.animationId != phase && anim.animationId != 0)
    {
      continue;
    }
    Rectangle srcRect = anim.srcRect;
    if (anim.frameCount > 1)
    {
      int w = anim.frameWidth > 0 ? anim.frameWidth : srcRect.width;
      srcRect.x += (int)(t / anim.frameDuration) % anim.frameCount * w;
    }
    Vector2 offset = { anim.offset.x / 16.0f * size, anim.offset.y / 16.0f * size * xScale };
    Vector2 scale = { srcRect.width / 16.0f * size, srcRect.height / 16.0f * size };
    
    if (flip)
    {
      srcRect.x += srcRect.width;
      srcRect.width = -srcRect.width;
      offset.x = scale.x - offset.x;
    }
    DrawBillboardPro(camera, spriteSheet, srcRect, position, up, scale, offset, 0, WHITE);
    // move the sprite slightly towards the camera to avoid z-fighting
    position = Vector3Add(position, Vector3Scale(Vector3Normalize(forward), -0.01f));  
  }
}

void TowerLoadAssets()
{
  towerModels[TOWER_TYPE_BASE] = LoadModel("data/keep.glb");
  towerModels[TOWER_TYPE_WALL] = LoadModel("data/wall-0000.glb");
}

void TowerInit()
{
  for (int i = 0; i < TOWER_MAX_COUNT; i++)
  {
    towers[i] = (Tower){0};
  }
  towerCount = 0;

  for (int i = 0; i < TOWER_TYPE_COUNT; i++)
  {
    if (towerModels[i].materials)
    {
      // assign the palette texture to the material of the model (0 is not used afaik)
      towerModels[i].materials[1].maps[MATERIAL_MAP_DIFFUSE].texture = palette;
    }
  }
}

static float TowerGetCooldown(Tower *tower)
{
  float cooldown = towerTypeConfigs[tower->towerType].cooldown;
  float maxUpgradeCooldown = towerTypeConfigs[tower->towerType].maxUpgradeCooldown;
  if (tower->upgradeState.speed > 0)
  {
    cooldown = Lerp(cooldown, maxUpgradeCooldown, tower->upgradeState.speed / (float)TOWER_MAX_STAGE);
  }
  return cooldown;
}

static void TowerGunUpdate(Tower *tower)
{
  TowerTypeConfig config = towerTypeConfigs[tower->towerType];
  if (tower->cooldown <= 0.0f)
  {
    Enemy *enemy = EnemyGetClosestToCastle(tower->x, tower->y, TowerGetRange(tower));
    if (enemy)
    {
      tower->cooldown = TowerGetCooldown(tower);
      // shoot the enemy; determine future position of the enemy
      float bulletSpeed = config.projectileSpeed;
      Vector2 velocity = enemy->simVelocity;
      Vector2 futurePosition = EnemyGetPosition(enemy, gameTime.time - enemy->startMovingTime, &velocity, 0);
      Vector2 towerPosition = {tower->x, tower->y};
      float eta = Vector2Distance(towerPosition, futurePosition) / bulletSpeed;
      for (int i = 0; i < 8; i++) {
        velocity = enemy->simVelocity;
        futurePosition = EnemyGetPosition(enemy, gameTime.time - enemy->startMovingTime + eta, &velocity, 0);
        float distance = Vector2Distance(towerPosition, futurePosition);
        float eta2 = distance / bulletSpeed;
        if (fabs(eta - eta2) < 0.01f) {
          break;
        }
        eta = (eta2 + eta) * 0.5f;
      }

      HitEffectConfig hitEffect = config.hitEffect;
      // apply damage upgrade to hit effect
      if (tower->upgradeState.damage > 0)
      {
        hitEffect.damage = Lerp(hitEffect.damage, hitEffect.maxUpgradeDamage, tower->upgradeState.damage / (float)TOWER_MAX_STAGE);
      }

      ProjectileTryAdd(config.projectileType, enemy, 
        (Vector3){towerPosition.x, 1.33f, towerPosition.y}, 
        (Vector3){futurePosition.x, 0.25f, futurePosition.y},
        bulletSpeed, hitEffect);
      enemy->futureDamage += hitEffect.damage;
      tower->lastTargetPosition = futurePosition;
    }
  }
  else
  {
    tower->cooldown -= gameTime.deltaTime;
  }
}

void TowerTypeSetData(uint8_t towerType, TowerTypeConfig *data)
{
  towerTypeConfigs[towerType] = *data;
}

Tower *TowerGetAt(int16_t x, int16_t y)
{
  for (int i = 0; i < towerCount; i++)
  {
    if (towers[i].x == x && towers[i].y == y && towers[i].towerType != TOWER_TYPE_NONE)
    {
      return &towers[i];
    }
  }
  return 0;
}

Tower *TowerGetByIndex(int index)
{
  if (index < 0 || index >= towerCount)
  {
    return 0;
  }
  return &towers[index];
}

Tower *TowerTryAdd(uint8_t towerType, int16_t x, int16_t y)
{
  if (towerCount >= TOWER_MAX_COUNT)
  {
    return 0;
  }

  Tower *tower = TowerGetAt(x, y);
  if (tower)
  {
    return 0;
  }

  tower = &towers[towerCount++];
  *tower = (Tower){
    .x = x,
    .y = y,
    .towerType = towerType,
    .cooldown = 0.0f,
    .damage = 0.0f,
  };
  return tower;
}

Tower *TowerGetByType(uint8_t towerType)
{
  for (int i = 0; i < towerCount; i++)
  {
    if (towers[i].towerType == towerType)
    {
      return &towers[i];
    }
  }
  return 0;
}

const char *TowerTypeGetName(uint8_t towerType)
{
  return towerTypeConfigs[towerType].name;
}

int TowerTypeGetCosts(uint8_t towerType)
{
  return towerTypeConfigs[towerType].cost;
}

float TowerGetMaxHealth(Tower *tower)
{
  return towerTypeConfigs[tower->towerType].maxHealth;
}

float TowerGetRange(Tower *tower)
{
  float range = towerTypeConfigs[tower->towerType].range;
  float maxUpgradeRange = towerTypeConfigs[tower->towerType].maxUpgradeRange;
  if (tower->upgradeState.range > 0)
  {
    range = Lerp(range, maxUpgradeRange, tower->upgradeState.range / (float)TOWER_MAX_STAGE);
  }
  return range;
}

void TowerUpdateAllRangeFade(Tower *selectedTower, float fadeoutTarget)
{
  // animate fade in and fade out of range drawing using framerate independent lerp
  float fadeLerp = 1.0f - powf(0.005f, gameTime.deltaTime);
  for (int i = 0; i < TOWER_MAX_COUNT; i++)
  {
    Tower *fadingTower = TowerGetByIndex(i);
    if (!fadingTower)
    {
      break;
    }
    float drawRangeTarget = fadingTower == selectedTower ? 1.0f : fadeoutTarget;
    fadingTower->drawRangeAlpha = Lerp(fadingTower->drawRangeAlpha, drawRangeTarget, fadeLerp);
  }
}

void TowerDrawRange(Tower *tower, float alpha)
{
  Color ringColor = (Color){255, 200, 100, 255};
  const int rings = 4;
  const float radiusOffset = 0.5f;
  const float animationSpeed = 2.0f;
  float animation = 1.0f - fmodf(gameTime.time * animationSpeed, 1.0f);
  float radius = TowerGetRange(tower);
  // base circle
  DrawCircle3D((Vector3){tower->x, 0.01f, tower->y}, radius, (Vector3){1, 0, 0}, 90, 
    Fade(ringColor, alpha));
  
  for (int i = 1; i < rings; i++)
  {
    float t = ((float)i + animation) / (float)rings;
    float r = Lerp(radius, radius - radiusOffset, t * t);
    float a = 1.0f - ((float)i + animation - 1) / (float) (rings - 1);
    if (i == 1)
    {
      // fade out the outermost ring
      a = animation;
    }
    a *= alpha;
    
    DrawCircle3D((Vector3){tower->x, 0.01f, tower->y}, r, (Vector3){1, 0, 0}, 90, 
      Fade(ringColor, a));
  }
}

void TowerDrawModel(Tower *tower)
{
  if (tower->towerType == TOWER_TYPE_NONE)
  {
    return;
  }

  if (tower->drawRangeAlpha > 2.0f/256.0f)
  {
   TowerDrawRange(tower, tower->drawRangeAlpha); 
  }

  switch (tower->towerType)
  {
  case TOWER_TYPE_ARCHER:
    {
      Vector2 screenPosTower = GetWorldToScreen((Vector3){tower->x, 0.0f, tower->y}, currentLevel->camera);
      Vector2 screenPosTarget = GetWorldToScreen((Vector3){tower->lastTargetPosition.x, 0.0f, tower->lastTargetPosition.y}, currentLevel->camera);
      DrawModel(towerModels[TOWER_TYPE_WALL], (Vector3){tower->x, 0.0f, tower->y}, 1.0f, WHITE);
      DrawSpriteUnit(archerUnit, (Vector3){tower->x, 1.0f, tower->y}, 0, screenPosTarget.x > screenPosTower.x, 
        tower->cooldown > 0.2f ? SPRITE_UNIT_PHASE_WEAPON_COOLDOWN : SPRITE_UNIT_PHASE_WEAPON_IDLE);
    }
    break;
  case TOWER_TYPE_BALLISTA:
    DrawCube((Vector3){tower->x, 0.5f, tower->y}, 1.0f, 1.0f, 1.0f, BROWN);
    break;
  case TOWER_TYPE_CATAPULT:
    DrawCube((Vector3){tower->x, 0.5f, tower->y}, 1.0f, 1.0f, 1.0f, DARKGRAY);
    break;
  default:
    if (towerModels[tower->towerType].materials)
    {
      DrawModel(towerModels[tower->towerType], (Vector3){tower->x, 0.0f, tower->y}, 1.0f, WHITE);
    } else {
      DrawCube((Vector3){tower->x, 0.5f, tower->y}, 1.0f, 1.0f, 1.0f, LIGHTGRAY);
    }
    break;
  }
}

void TowerDrawAll()
{
  for (int i = 0; i < towerCount; i++)
  {
    TowerDrawModel(&towers[i]);
  }
}

void TowerUpdate()
{
  for (int i = 0; i < towerCount; i++)
  {
    Tower *tower = &towers[i];
    switch (tower->towerType)
    {
    case TOWER_TYPE_CATAPULT:
    case TOWER_TYPE_BALLISTA:
    case TOWER_TYPE_ARCHER:
      TowerGunUpdate(tower);
      break;
    }
  }
}

void TowerDrawAllHealthBars(Camera3D camera)
{
  for (int i = 0; i < towerCount; i++)
  {
    Tower *tower = &towers[i];
    if (tower->towerType == TOWER_TYPE_NONE || tower->damage <= 0.0f)
    {
      continue;
    }
    
    Vector3 position = (Vector3){tower->x, 0.5f, tower->y};
    float maxHealth = TowerGetMaxHealth(tower);
    float health = maxHealth - tower->damage;
    float healthRatio = health / maxHealth;
    
    DrawHealthBar(camera, position, Vector2Zero(), healthRatio, GREEN, 35.0f);
  }
}