#include "td_main.h"
#include <raymath.h>
#include <rlgl.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#ifdef PLATFORM_WEB
#include <emscripten/emscripten.h>
#else
#define EMSCRIPTEN_KEEPALIVE
#endif

//# Variables
Font gameFontNormal = {0};
GUIState guiState = {0};
GameTime gameTime = {
  .fixedDeltaTime = 1.0f / 60.0f,
};

Model floorTileAModel = {0};
Model floorTileBModel = {0};
Model treeModel[2] = {0};
Model firTreeModel[2] = {0};
Model rockModels[5] = {0};
Model grassPatchModel[1] = {0};

Model pathArrowModel = {0};
Model greenArrowModel = {0};

Texture2D palette, spriteSheet;

NPatchInfo uiPanelPatch = {
  .layout = NPATCH_NINE_PATCH,
  .source = {145, 1, 46, 46},
  .top = 18, .bottom = 18,
  .left = 16, .right = 16
};
NPatchInfo uiButtonNormal = {
  .layout = NPATCH_NINE_PATCH,
  .source = {193, 1, 32, 20},
  .top = 7, .bottom = 7,
  .left = 10, .right = 10
};
NPatchInfo uiButtonDisabled = {
  .layout = NPATCH_NINE_PATCH,
  .source = {193, 22, 32, 20},
  .top = 7, .bottom = 7,
  .left = 10, .right = 10
};
NPatchInfo uiButtonHovered = {
  .layout = NPATCH_NINE_PATCH,
  .source = {193, 43, 32, 20},
  .top = 7, .bottom = 7,
  .left = 10, .right = 10
};
NPatchInfo uiButtonPressed = {
  .layout = NPATCH_NINE_PATCH,
  .source = {193, 64, 32, 20},
  .top = 7, .bottom = 7,
  .left = 10, .right = 10
};
Rectangle uiDiamondMarker = {145, 48, 15, 15};

Level loadedLevels[32] = {0};
Level levels[] = {
  [0] = {
    .state = LEVEL_STATE_BUILDING,
    .initialGold = 500,
    .waves[0] = {
      .enemyType = ENEMY_TYPE_SHIELD,
      .wave = 0,
      .count = 1,
      .interval = 2.5f,
      .delay = 1.0f,
      .spawnPosition = {2, 6},
    },
    .waves[1] = {
      .enemyType = ENEMY_TYPE_RUNNER,
      .wave = 0,
      .count = 5,
      .interval = 0.5f,
      .delay = 1.0f,
      .spawnPosition = {-2, 6},
    },
    .waves[2] = {
      .enemyType = ENEMY_TYPE_SHIELD,
      .wave = 1,
      .count = 20,
      .interval = 1.5f,
      .delay = 1.0f,
      .spawnPosition = {0, 6},
    },
    .waves[3] = {
      .enemyType = ENEMY_TYPE_MINION,
      .wave = 2,
      .count = 30,
      .interval = 1.2f,
      .delay = 1.0f,
      .spawnPosition = {2, 6},
    },
    .waves[4] = {
      .enemyType = ENEMY_TYPE_BOSS,
      .wave = 2,
      .count = 2,
      .interval = 5.0f,
      .delay = 2.0f,
      .spawnPosition = {-2, 4},
    }
  },
};

Level *currentLevel = levels;

void DrawBoxedText(const char *text, int x, int y, int width, int height, float alignX, float alignY, Color textColor);
int LoadConfig();

void DrawTitleText(const char *text, int anchorX, float alignX, Color color)
{
  int textWidth = MeasureTextEx(gameFontNormal, text, gameFontNormal.baseSize, 0).x;
  int panelWidth = textWidth + 40;
  int posX = anchorX - panelWidth * alignX;
  int textOffset = 20;
  DrawTextureNPatch(spriteSheet, uiPanelPatch, (Rectangle){ posX, -10, panelWidth, 40}, Vector2Zero(), 0, WHITE);
  DrawTextEx(gameFontNormal, text, (Vector2){posX + textOffset + 1, 6 + 1}, gameFontNormal.baseSize, 0, BLACK);
  DrawTextEx(gameFontNormal, text, (Vector2){posX + textOffset, 6}, gameFontNormal.baseSize, 0, color);
}

void DrawTitle(const char *text)
{
  DrawTitleText(text, GetScreenWidth() / 2, 0.5f, WHITE);
}

//# Game

static Model LoadGLBModel(char *filename)
{
  Model model = LoadModel(TextFormat("data/%s.glb",filename));
  for (int i = 0; i < model.materialCount; i++)
  {
    model.materials[i].maps[MATERIAL_MAP_DIFFUSE].texture = palette;
  }
  return model;
}

void LoadAssets()
{
  // load a sprite sheet that contains all units
  spriteSheet = LoadTexture("data/spritesheet.png");
  SetTextureFilter(spriteSheet, TEXTURE_FILTER_BILINEAR);

  // we'll use a palette texture to colorize the all buildings and environment art
  palette = LoadTexture("data/palette.png");
  // The texture uses gradients on very small space, so we'll enable bilinear filtering
  SetTextureFilter(palette, TEXTURE_FILTER_BILINEAR);

  gameFontNormal = LoadFont("data/alagard.png");

  floorTileAModel = LoadGLBModel("floor-tile-a");
  floorTileBModel = LoadGLBModel("floor-tile-b");
  treeModel[0] = LoadGLBModel("leaftree-large-1-a");
  treeModel[1] = LoadGLBModel("leaftree-large-1-b");
  firTreeModel[0] = LoadGLBModel("firtree-1-a");
  firTreeModel[1] = LoadGLBModel("firtree-1-b");
  rockModels[0] = LoadGLBModel("rock-1");
  rockModels[1] = LoadGLBModel("rock-2");
  rockModels[2] = LoadGLBModel("rock-3");
  rockModels[3] = LoadGLBModel("rock-4");
  rockModels[4] = LoadGLBModel("rock-5");
  grassPatchModel[0] = LoadGLBModel("grass-patch-1");

  pathArrowModel = LoadGLBModel("direction-arrow-x");
  greenArrowModel = LoadGLBModel("green-arrow");

  TowerLoadAssets();
}

void InitLevel(Level *level)
{
  level->seed = (int)(GetTime() * 100.0f);

  TowerInit();
  EnemyInit();
  ProjectileInit();
  ParticleInit();
  TowerTryAdd(TOWER_TYPE_BASE, 0, 0);

  level->placementMode = 0;
  level->state = LEVEL_STATE_BUILDING;
  level->nextState = LEVEL_STATE_NONE;
  level->playerGold = level->initialGold;
  level->currentWave = 0;
  level->placementX = -1;
  level->placementY = 0;

  Camera *camera = &level->camera;
  camera->position = (Vector3){4.0f, 8.0f, 8.0f};
  camera->target = (Vector3){0.0f, 0.0f, 0.0f};
  camera->up = (Vector3){0.0f, 1.0f, 0.0f};
  camera->fovy = 11.5f;
  camera->projection = CAMERA_ORTHOGRAPHIC;
}

void DrawLevelHud(Level *level)
{
  const char *text = TextFormat("Gold: %d", level->playerGold);
  DrawTitleText(text, GetScreenWidth() - 10, 1.0f, YELLOW);
}

void DrawLevelReportLostWave(Level *level)
{
  BeginMode3D(level->camera);
  DrawLevelGround(level);
  TowerUpdateAllRangeFade(0, 0.0f);
  TowerDrawAll();
  EnemyDraw();
  ProjectileDraw();
  ParticleDraw();
  guiState.isBlocked = 0;
  EndMode3D();

  TowerDrawAllHealthBars(level->camera);

  DrawTitle("Wave lost");
  
  if (Button("Reset level", 20, GetScreenHeight() - 40, 160, 30, 0))
  {
    level->nextState = LEVEL_STATE_RESET;
  }
}

int HasLevelNextWave(Level *level)
{
  for (int i = 0; i < 10; i++)
  {
    EnemyWave *wave = &level->waves[i];
    if (wave->wave == level->currentWave)
    {
      return 1;
    }
  }
  return 0;
}

void DrawLevelReportWonWave(Level *level)
{
  BeginMode3D(level->camera);
  DrawLevelGround(level);
  TowerUpdateAllRangeFade(0, 0.0f);
  TowerDrawAll();
  EnemyDraw();
  ProjectileDraw();
  ParticleDraw();
  guiState.isBlocked = 0;
  EndMode3D();

  TowerDrawAllHealthBars(level->camera);

  DrawTitle("Wave won");


  if (Button("Reset level", 20, GetScreenHeight() - 40, 160, 30, 0))
  {
    level->nextState = LEVEL_STATE_RESET;
  }

  if (HasLevelNextWave(level))
  {
    if (Button("Prepare for next wave", GetScreenWidth() - 300, GetScreenHeight() - 40, 300, 30, 0))
    {
      level->nextState = LEVEL_STATE_BUILDING;
    }
  }
  else {
    if (Button("Level won", GetScreenWidth() - 300, GetScreenHeight() - 40, 300, 30, 0))
    {
      level->nextState = LEVEL_STATE_WON_LEVEL;
    }
  }
}

int DrawBuildingBuildButton(Level *level, int x, int y, int width, int height, uint8_t towerType, const char *name)
{
  static ButtonState buttonStates[8] = {0};
  int cost = TowerTypeGetCosts(towerType);
  const char *text = TextFormat("%s: %d", name, cost);
  buttonStates[towerType].isSelected = level->placementMode == towerType;
  buttonStates[towerType].isDisabled = level->playerGold < cost;
  if (Button(text, x, y, width, height, &buttonStates[towerType]))
  {
    level->placementMode = buttonStates[towerType].isSelected ? 0 : towerType;
    level->nextState = LEVEL_STATE_BUILDING_PLACEMENT;
    return 1;
  }
  return 0;
}

float GetRandomFloat(float min, float max)
{
  int random = GetRandomValue(0, 0xfffffff);
  return ((float)random / (float)0xfffffff) * (max - min) + min;
}

void DrawLevelGround(Level *level)
{
  // draw checkerboard ground pattern
  for (int x = -5; x <= 5; x += 1)
  {
    for (int y = -5; y <= 5; y += 1)
    {
      Model *model = (x + y) % 2 == 0 ? &floorTileAModel : &floorTileBModel;
      DrawModel(*model, (Vector3){x, 0.0f, y}, 1.0f, WHITE);
    }
  }

  int oldSeed = GetRandomValue(0, 0xfffffff);
  SetRandomSeed(level->seed);
  // increase probability for trees via duplicated entries
  Model borderModels[64];
  int maxRockCount = GetRandomValue(2, 6);
  int maxTreeCount = GetRandomValue(10, 20);
  int maxFirTreeCount = GetRandomValue(5, 10);
  int maxLeafTreeCount = maxTreeCount - maxFirTreeCount;
  int grassPatchCount = GetRandomValue(5, 30);

  int modelCount = 0;
  for (int i = 0; i < maxRockCount && modelCount < 63; i++)
  {
    borderModels[modelCount++] = rockModels[GetRandomValue(0, 5)];
  }
  for (int i = 0; i < maxLeafTreeCount && modelCount < 63; i++)
  {
    borderModels[modelCount++] = treeModel[GetRandomValue(0, 1)];
  }
  for (int i = 0; i < maxFirTreeCount && modelCount < 63; i++)
  {
    borderModels[modelCount++] = firTreeModel[GetRandomValue(0, 1)];
  }
  for (int i = 0; i < grassPatchCount && modelCount < 63; i++)
  {
    borderModels[modelCount++] = grassPatchModel[0];
  }

  // draw some objects around the border of the map
  Vector3 up = {0, 1, 0};
  // a pseudo random number generator to get the same result every time
  const float wiggle = 0.75f;
  const int layerCount = 3;
  for (int layer = 0; layer <= layerCount; layer++)
  {
    int layerPos = 6 + layer;
    Model *selectedModels = borderModels;
    int selectedModelCount = modelCount;
    if (layer == 0)
    {
      selectedModels = grassPatchModel;
      selectedModelCount = 1;
    }
    for (int x = -6 - layer; x <= 6 + layer; x += 1)
    {
      DrawModelEx(selectedModels[GetRandomValue(0, selectedModelCount - 1)], 
        (Vector3){x + GetRandomFloat(0.0f, wiggle), 0.0f, -layerPos + GetRandomFloat(0.0f, wiggle)}, 
        up, GetRandomFloat(0.0f, 360), Vector3One(), WHITE);
      DrawModelEx(selectedModels[GetRandomValue(0, selectedModelCount - 1)], 
        (Vector3){x + GetRandomFloat(0.0f, wiggle), 0.0f, layerPos + GetRandomFloat(0.0f, wiggle)}, 
        up, GetRandomFloat(0.0f, 360), Vector3One(), WHITE);
    }

    for (int z = -5 - layer; z <= 5 + layer; z += 1)
    {
      DrawModelEx(selectedModels[GetRandomValue(0, selectedModelCount - 1)], 
        (Vector3){-layerPos + GetRandomFloat(0.0f, wiggle), 0.0f, z + GetRandomFloat(0.0f, wiggle)}, 
        up, GetRandomFloat(0.0f, 360), Vector3One(), WHITE);
      DrawModelEx(selectedModels[GetRandomValue(0, selectedModelCount - 1)], 
        (Vector3){layerPos + GetRandomFloat(0.0f, wiggle), 0.0f, z + GetRandomFloat(0.0f, wiggle)}, 
        up, GetRandomFloat(0.0f, 360), Vector3One(), WHITE);
    }
  }

  SetRandomSeed(oldSeed);
}

void DrawEnemyPath(Level *level, Color arrowColor)
{
  const int castleX = 0, castleY = 0;
  const int maxWaypointCount = 200;
  const float timeStep = 1.0f;
  Vector3 arrowScale = {0.75f, 0.75f, 0.75f};

  // we start with a time offset to simulate the path, 
  // this way the arrows are animated in a forward moving direction
  // The time is wrapped around the time step to get a smooth animation
  float timeOffset = fmodf(GetTime(), timeStep);

  for (int i = 0; i < ENEMY_MAX_WAVE_COUNT; i++)
  {
    EnemyWave *wave = &level->waves[i];
    if (wave->wave != level->currentWave)
    {
      continue;
    }

    // use this dummy enemy to simulate the path
    Enemy dummy = {
      .enemyType = ENEMY_TYPE_MINION,
      .simPosition = (Vector2){wave->spawnPosition.x, wave->spawnPosition.y},
      .nextX = wave->spawnPosition.x,
      .nextY = wave->spawnPosition.y,
      .currentX = wave->spawnPosition.x,
      .currentY = wave->spawnPosition.y,
    };

    float deltaTime = timeOffset;
    for (int j = 0; j < maxWaypointCount; j++)
    {
      int waypointPassedCount = 0;
      Vector2 pos = EnemyGetPosition(&dummy, deltaTime, &dummy.simVelocity, &waypointPassedCount);
      // after the initial variable starting offset, we use a fixed time step
      deltaTime = timeStep;
      dummy.simPosition = pos;

      // Update the dummy's position just like we do in the regular enemy update loop
      for (int k = 0; k < waypointPassedCount; k++)
      {
        dummy.currentX = dummy.nextX;
        dummy.currentY = dummy.nextY;
        if (EnemyGetNextPosition(dummy.currentX, dummy.currentY, &dummy.nextX, &dummy.nextY) &&
          Vector2DistanceSqr(dummy.simPosition, (Vector2){castleX, castleY}) <= 0.25f * 0.25f)
        {
          break;
        }
      }
      if (Vector2DistanceSqr(dummy.simPosition, (Vector2){castleX, castleY}) <= 0.25f * 0.25f)
      {
        break;
      }
      
      // get the angle we need to rotate the arrow model. The velocity is just fine for this.
      float angle = atan2f(dummy.simVelocity.x, dummy.simVelocity.y) * RAD2DEG - 90.0f;
      DrawModelEx(pathArrowModel, (Vector3){pos.x, 0.15f, pos.y}, (Vector3){0, 1, 0}, angle, arrowScale, arrowColor);
    }
  }
}

void DrawEnemyPaths(Level *level)
{
  // disable depth testing for the path arrows
  // flush the 3D batch to draw the arrows on top of everything
  rlDrawRenderBatchActive();
  rlDisableDepthTest();
  DrawEnemyPath(level, (Color){64, 64, 64, 160});

  rlDrawRenderBatchActive();
  rlEnableDepthTest();
  DrawEnemyPath(level, WHITE);
}

static void SimulateTowerPlacementBehavior(Level *level, float towerFloatHeight, float mapX, float mapY)
{
  float dt = gameTime.fixedDeltaTime;
  // smooth transition for the placement position using exponential decay
  const float lambda = 15.0f;
  float factor = 1.0f - expf(-lambda * dt);

  float damping = 0.5f;
  float springStiffness = 300.0f;
  float springDecay = 95.0f;
  float minHeight = 0.35f;

  if (level->placementPhase == PLACEMENT_PHASE_STARTING)
  {
    damping = 1.0f;
    springDecay = 90.0f;
    springStiffness = 100.0f;
    minHeight = 0.70f;
  }

  for (int i = 0; i < gameTime.fixedStepCount; i++)
  {
    level->placementTransitionPosition = 
      Vector2Lerp(
        level->placementTransitionPosition, 
        (Vector2){mapX, mapY}, factor);

    // draw the spring position for debugging the spring simulation
    // first step: stiff spring, no simulation
    Vector3 worldPlacementPosition = (Vector3){
      level->placementTransitionPosition.x, towerFloatHeight, level->placementTransitionPosition.y};
    Vector3 springTargetPosition = (Vector3){
      worldPlacementPosition.x, 1.0f + towerFloatHeight, worldPlacementPosition.z};
    // consider the current velocity to predict the future position in order to dampen
    // the spring simulation. Longer prediction times will result in more damping
    Vector3 predictedSpringPosition = Vector3Add(level->placementTowerSpring.position, 
      Vector3Scale(level->placementTowerSpring.velocity, dt * damping));
    Vector3 springPointDelta = Vector3Subtract(springTargetPosition, predictedSpringPosition);
    Vector3 velocityChange = Vector3Scale(springPointDelta, dt * springStiffness);
    // decay velocity of the upright forcing spring
    // This force acts like a 2nd spring that pulls the tip upright into the air above the
    // base position
    level->placementTowerSpring.velocity = Vector3Scale(level->placementTowerSpring.velocity, 1.0f - expf(-springDecay * dt));
    level->placementTowerSpring.velocity = Vector3Add(level->placementTowerSpring.velocity, velocityChange);

    // calculate length of the spring to calculate the force to apply to the placementTowerSpring position
    // we use a simple spring model with a rest length of 1.0f
    Vector3 springDelta = Vector3Subtract(worldPlacementPosition, level->placementTowerSpring.position);
    float springLength = Vector3Length(springDelta);
    float springForce = (springLength - 1.0f) * springStiffness;
    Vector3 springForceVector = Vector3Normalize(springDelta);
    springForceVector = Vector3Scale(springForceVector, springForce);
    level->placementTowerSpring.velocity = Vector3Add(level->placementTowerSpring.velocity, 
      Vector3Scale(springForceVector, dt));

    level->placementTowerSpring.position = Vector3Add(level->placementTowerSpring.position, 
      Vector3Scale(level->placementTowerSpring.velocity, dt));
    if (level->placementTowerSpring.position.y < minHeight + towerFloatHeight)
    {
      level->placementTowerSpring.velocity.y *= -1.0f;
      level->placementTowerSpring.position.y = fmaxf(level->placementTowerSpring.position.y, minHeight + towerFloatHeight);
    }
  }
}

void DrawLevelBuildingPlacementState(Level *level)
{
  const float placementDuration = 0.5f;

  level->placementTimer += gameTime.deltaTime;
  if (level->placementTimer > 1.0f && level->placementPhase == PLACEMENT_PHASE_STARTING)
  {
    level->placementPhase = PLACEMENT_PHASE_MOVING;
    level->placementTimer = 0.0f;
  }

  BeginMode3D(level->camera);
  DrawLevelGround(level);

  int blockedCellCount = 0;
  Vector2 blockedCells[1];
  Ray ray = GetScreenToWorldRay(GetMousePosition(), level->camera);
  float planeDistance = ray.position.y / -ray.direction.y;
  float planeX = ray.direction.x * planeDistance + ray.position.x;
  float planeY = ray.direction.z * planeDistance + ray.position.z;
  int16_t mapX = (int16_t)floorf(planeX + 0.5f);
  int16_t mapY = (int16_t)floorf(planeY + 0.5f);
  if (level->placementPhase == PLACEMENT_PHASE_MOVING && 
    level->placementMode && !guiState.isBlocked && 
    mapX >= -5 && mapX <= 5 && mapY >= -5 && mapY <= 5 && IsMouseButtonDown(MOUSE_LEFT_BUTTON))
  {
    level->placementX = mapX;
    level->placementY = mapY;
  }
  else
  {
    mapX = level->placementX;
    mapY = level->placementY;
  }
  blockedCells[blockedCellCount++] = (Vector2){mapX, mapY};
  PathFindingMapUpdate(blockedCellCount, blockedCells);

  TowerUpdateAllRangeFade(0, 0.0f);
  TowerDrawAll();
  EnemyDraw();
  ProjectileDraw();
  ParticleDraw();
  DrawEnemyPaths(level);

  // let the tower float up and down. Consider this height in the spring simulation as well
  float towerFloatHeight = sinf(gameTime.time * 4.0f) * 0.2f + 0.3f;

  if (level->placementPhase == PLACEMENT_PHASE_PLACING)
  {
    // The bouncing spring needs a bit of outro time to look nice and complete. 
    // So we scale the time so that the first 2/3rd of the placing phase handles the motion
    // and the last 1/3rd is the outro physics (bouncing)
    float t = fminf(1.0f, level->placementTimer / placementDuration * 1.5f);
    // let towerFloatHeight describe parabola curve, starting at towerFloatHeight and ending at 0
    float linearBlendHeight = (1.0f - t) * towerFloatHeight;
    float parabola = (1.0f - ((t - 0.5f) * (t - 0.5f) * 4.0f)) * 2.0f;
    towerFloatHeight = linearBlendHeight + parabola;
  }

  SimulateTowerPlacementBehavior(level, towerFloatHeight, mapX, mapY);
  
  rlPushMatrix();
  rlTranslatef(level->placementTransitionPosition.x, 0, level->placementTransitionPosition.y);

  // calculate x and z rotation to align the model with the spring
  Vector3 position = {level->placementTransitionPosition.x, towerFloatHeight, level->placementTransitionPosition.y};
  Vector3 towerUp = Vector3Subtract(level->placementTowerSpring.position, position);
  Vector3 rotationAxis = Vector3CrossProduct(towerUp, (Vector3){0, 1, 0});
  float angle = acosf(Vector3DotProduct(towerUp, (Vector3){0, 1, 0}) / Vector3Length(towerUp)) * RAD2DEG;
  float springLength = Vector3Length(towerUp);
  float towerStretch = fminf(fmaxf(springLength, 0.5f), 4.5f);
  float towerSquash = 1.0f / towerStretch;

  Tower dummy = {
    .towerType = level->placementMode,
  };
  
  float rangeAlpha = fminf(1.0f, level->placementTimer / placementDuration);
  if (level->placementPhase == PLACEMENT_PHASE_PLACING)
  {
    rangeAlpha = 1.0f - rangeAlpha;
  }
  else if (level->placementPhase == PLACEMENT_PHASE_MOVING)
  {
    rangeAlpha = 1.0f;
  }

  TowerDrawRange(&dummy, rangeAlpha);
  
  rlPushMatrix();
  rlTranslatef(0.0f, towerFloatHeight, 0.0f);
  
  rlRotatef(-angle, rotationAxis.x, rotationAxis.y, rotationAxis.z);
  rlScalef(towerSquash, towerStretch, towerSquash);
  TowerDrawModel(&dummy);
  rlPopMatrix();


  // draw a shadow for the tower
  float umbrasize = 0.8 + sqrtf(towerFloatHeight);
  DrawCube((Vector3){0.0f, 0.05f, 0.0f}, umbrasize, 0.0f, umbrasize, (Color){0, 0, 0, 32});
  DrawCube((Vector3){0.0f, 0.075f, 0.0f}, 0.85f, 0.0f, 0.85f, (Color){0, 0, 0, 64});


  float bounce = sinf(gameTime.time * 8.0f) * 0.5f + 0.5f;
  float offset = fmaxf(0.0f, bounce - 0.4f) * 0.35f + 0.7f;
  float squeeze = -fminf(0.0f, bounce - 0.4f) * 0.7f + 0.7f;
  float stretch = fmaxf(0.0f, bounce - 0.3f) * 0.5f + 0.8f;
  
  DrawModelEx(greenArrowModel, (Vector3){ offset, 0.0f,  0.0f}, (Vector3){0, 1, 0}, 90, (Vector3){squeeze, 1.0f, stretch}, WHITE);
  DrawModelEx(greenArrowModel, (Vector3){-offset, 0.0f,  0.0f}, (Vector3){0, 1, 0}, 270, (Vector3){squeeze, 1.0f, stretch}, WHITE);
  DrawModelEx(greenArrowModel, (Vector3){ 0.0f, 0.0f,  offset}, (Vector3){0, 1, 0}, 0, (Vector3){squeeze, 1.0f, stretch}, WHITE);
  DrawModelEx(greenArrowModel, (Vector3){ 0.0f, 0.0f, -offset}, (Vector3){0, 1, 0}, 180, (Vector3){squeeze, 1.0f, stretch}, WHITE);
  rlPopMatrix();

  guiState.isBlocked = 0;

  EndMode3D();

  TowerDrawAllHealthBars(level->camera);

  if (level->placementPhase == PLACEMENT_PHASE_PLACING)
  {
    if (level->placementTimer > placementDuration)
    {
        Tower *tower = TowerTryAdd(level->placementMode, mapX, mapY);
        // testing repairing
        tower->damage = 2.5f;
        level->playerGold -= TowerTypeGetCosts(level->placementMode);
        level->nextState = LEVEL_STATE_BUILDING;
        level->placementMode = TOWER_TYPE_NONE;
    }
  }
  else
  {   
    if (Button("Cancel", 20, GetScreenHeight() - 40, 160, 30, 0))
    {
      level->nextState = LEVEL_STATE_BUILDING;
      level->placementMode = TOWER_TYPE_NONE;
      TraceLog(LOG_INFO, "Cancel building");
    }
    
    if (TowerGetAt(mapX, mapY) == 0 &&  Button("Build", GetScreenWidth() - 180, GetScreenHeight() - 40, 160, 30, 0))
    {
      level->placementPhase = PLACEMENT_PHASE_PLACING;
      level->placementTimer = 0.0f;
    }
  }
}

enum ContextMenuType
{
  CONTEXT_MENU_TYPE_MAIN,
  CONTEXT_MENU_TYPE_SELL_CONFIRM,
  CONTEXT_MENU_TYPE_UPGRADE,
};

enum UpgradeType
{
  UPGRADE_TYPE_SPEED,
  UPGRADE_TYPE_DAMAGE,
  UPGRADE_TYPE_RANGE,
};

typedef struct ContextMenuArgs
{
  void *data;
  uint8_t uint8;
  int32_t int32;
  Tower *tower;
} ContextMenuArgs;

int OnContextMenuBuild(Level *level, ContextMenuArgs *data)
{
  uint8_t towerType = data->uint8;
  level->placementMode = towerType;
  level->nextState = LEVEL_STATE_BUILDING_PLACEMENT;
  return 1;
}

int OnContextMenuSellConfirm(Level *level, ContextMenuArgs *data)
{
  Tower *tower = data->tower;
  int gold = data->int32;
  level->playerGold += gold;
  tower->towerType = TOWER_TYPE_NONE;
  return 1;
}

int OnContextMenuSellCancel(Level *level, ContextMenuArgs *data)
{
  return 1;
}

int OnContextMenuSell(Level *level, ContextMenuArgs *data)
{
  level->placementContextMenuType = CONTEXT_MENU_TYPE_SELL_CONFIRM;
  return 0;
}

int OnContextMenuDoUpgrade(Level *level, ContextMenuArgs *data)
{
  Tower *tower = data->tower;
  switch (data->uint8)
  {
    case UPGRADE_TYPE_SPEED: tower->upgradeState.speed++; break;
    case UPGRADE_TYPE_DAMAGE: tower->upgradeState.damage++; break;
    case UPGRADE_TYPE_RANGE: tower->upgradeState.range++; break;
  }
  level->playerGold -= data->int32;
  return 0;
}

int OnContextMenuUpgrade(Level *level, ContextMenuArgs *data)
{
  level->placementContextMenuType = CONTEXT_MENU_TYPE_UPGRADE;
  return 0;
}

int OnContextMenuRepair(Level *level, ContextMenuArgs *data)
{
  Tower *tower = data->tower;
  if (level->playerGold >= 1)
  {
    level->playerGold -= 1;
    tower->damage = fmaxf(0.0f, tower->damage - 1.0f);
  }
  return tower->damage == 0.0f;
}

typedef struct ContextMenuItem
{
  uint8_t index;
  char text[24];
  float alignX;
  int (*action)(Level*, ContextMenuArgs*);
  void *data;
  ContextMenuArgs args;
  ButtonState buttonState;
} ContextMenuItem;

ContextMenuItem ContextMenuItemText(uint8_t index, const char *text, float alignX)
{
  ContextMenuItem item = {.index = index, .alignX = alignX};
  strncpy(item.text, text, 23);
  item.text[23] = 0;
  return item;
}

ContextMenuItem ContextMenuItemButton(uint8_t index, const char *text, int (*action)(Level*, ContextMenuArgs*), ContextMenuArgs args)
{
  ContextMenuItem item = {.index = index, .action = action, .args = args};
  strncpy(item.text, text, 23);
  item.text[23] = 0;
  return item;
}

int DrawContextMenu(Level *level, Vector2 anchorLow, Vector2 anchorHigh, int width, ContextMenuItem *menus)
{
  const int itemHeight = 28;
  const int itemSpacing = 1;
  const int padding = 8;
  int itemCount = 0;
  for (int i = 0; menus[i].text[0]; i++)
  {
    itemCount = itemCount > menus[i].index ? itemCount : menus[i].index;
  }

  Rectangle contextMenu = {0, 0, width, 
    (itemHeight + itemSpacing) * (itemCount + 1) + itemSpacing + padding * 2};
  
  Vector2 anchor = anchorHigh.y - 30 > contextMenu.height ? anchorHigh : anchorLow;
  float anchorPivotY = anchorHigh.y - 30 > contextMenu.height ? 1.0f : 0.0f;
  
  contextMenu.x = anchor.x - contextMenu.width * 0.5f;
  contextMenu.y = anchor.y - contextMenu.height * anchorPivotY;
  contextMenu.x = fmaxf(0, fminf(GetScreenWidth() - contextMenu.width, contextMenu.x));
  contextMenu.y = fmaxf(0, fminf(GetScreenHeight() - contextMenu.height, contextMenu.y));

  DrawTextureNPatch(spriteSheet, uiPanelPatch, contextMenu, Vector2Zero(), 0, WHITE);
  DrawTextureRec(spriteSheet, uiDiamondMarker, (Vector2){anchor.x - uiDiamondMarker.width / 2, anchor.y - uiDiamondMarker.height / 2}, WHITE);
  const int itemX = contextMenu.x + itemSpacing;
  const int itemWidth = contextMenu.width - itemSpacing * 2;
  #define ITEM_Y(idx) (contextMenu.y + itemSpacing + (itemHeight + itemSpacing) * idx + padding)
  #define ITEM_RECT(idx, marginLR) itemX + (marginLR) + padding, ITEM_Y(idx), itemWidth - (marginLR) * 2 - padding * 2, itemHeight
  int status = 0;
  for (int i = 0; menus[i].text[0]; i++)
  {
    if (menus[i].action)
    {
      if (Button(menus[i].text, ITEM_RECT(menus[i].index, 0), &menus[i].buttonState))
      {
        status = menus[i].action(level, &menus[i].args);
      }
    }
    else
    {
      DrawBoxedText(menus[i].text, ITEM_RECT(menus[i].index, 0), menus[i].alignX, 0.5f, WHITE);
    }
  }

  if (IsMouseButtonPressed(MOUSE_LEFT_BUTTON) && !CheckCollisionPointRec(GetMousePosition(), contextMenu))
  {
    return 1;
  }

  return status;
}

void DrawLevelBuildingStateMainContextMenu(Level *level, Tower *tower, int32_t sellValue, float hp, float maxHitpoints, Vector2 anchorLow, Vector2 anchorHigh)
{
  ContextMenuItem menu[12] = {0};
  int menuCount = 0;
  int menuIndex = 0;
  if (tower)
  {

    if (tower) {
      menu[menuCount++] = ContextMenuItemText(menuIndex++, TowerTypeGetName(tower->towerType), 0.5f);
    }

    // two texts, same line
    menu[menuCount++] = ContextMenuItemText(menuIndex, "HP:", 0.0f);
    menu[menuCount++] = ContextMenuItemText(menuIndex++, TextFormat("%.1f / %.1f", hp, maxHitpoints), 1.0f);

    if (tower->towerType != TOWER_TYPE_BASE)
    {
      menu[menuCount++] = ContextMenuItemButton(menuIndex++, "Upgrade", OnContextMenuUpgrade, 
        (ContextMenuArgs){.tower = tower});
    }

    if (tower->towerType != TOWER_TYPE_BASE)
    {
      
      menu[menuCount++] = ContextMenuItemButton(menuIndex++, "Sell", OnContextMenuSell, 
        (ContextMenuArgs){.tower = tower, .int32 = sellValue});
    }
    if (tower->towerType != TOWER_TYPE_BASE && tower->damage > 0 && level->playerGold >= 1)
    {
      menu[menuCount++] = ContextMenuItemButton(menuIndex++, "Repair 1 (1G)", OnContextMenuRepair, 
        (ContextMenuArgs){.tower = tower});
    }
  }
  else
  {
    menu[menuCount] = ContextMenuItemButton(menuIndex++, 
      TextFormat("Wall: %dG", TowerTypeGetCosts(TOWER_TYPE_WALL)),
      OnContextMenuBuild, (ContextMenuArgs){ .uint8 = (uint8_t) TOWER_TYPE_WALL});
    menu[menuCount++].buttonState.isDisabled = level->playerGold < TowerTypeGetCosts(TOWER_TYPE_WALL);

    menu[menuCount] = ContextMenuItemButton(menuIndex++, 
      TextFormat("Archer: %dG", TowerTypeGetCosts(TOWER_TYPE_ARCHER)),
      OnContextMenuBuild, (ContextMenuArgs){ .uint8 = (uint8_t) TOWER_TYPE_ARCHER});
    menu[menuCount++].buttonState.isDisabled = level->playerGold < TowerTypeGetCosts(TOWER_TYPE_ARCHER);

    menu[menuCount] = ContextMenuItemButton(menuIndex++, 
      TextFormat("Ballista: %dG", TowerTypeGetCosts(TOWER_TYPE_BALLISTA)),
      OnContextMenuBuild, (ContextMenuArgs){ .uint8 = (uint8_t) TOWER_TYPE_BALLISTA});
    menu[menuCount++].buttonState.isDisabled = level->playerGold < TowerTypeGetCosts(TOWER_TYPE_BALLISTA);

    menu[menuCount] = ContextMenuItemButton(menuIndex++, 
      TextFormat("Catapult: %dG", TowerTypeGetCosts(TOWER_TYPE_CATAPULT)),
      OnContextMenuBuild, (ContextMenuArgs){ .uint8 = (uint8_t) TOWER_TYPE_CATAPULT});
    menu[menuCount++].buttonState.isDisabled = level->playerGold < TowerTypeGetCosts(TOWER_TYPE_CATAPULT);
  }
  
  if (DrawContextMenu(level, anchorLow, anchorHigh, 150, menu))
  {
    level->placementContextMenuStatus = -1;
  }
}

void DrawLevelBuildingState(Level *level)
{
  // when the context menu is not active, we update the placement position
  if (level->placementContextMenuStatus == 0)
  {
    Ray ray = GetScreenToWorldRay(GetMousePosition(), level->camera);
    float hitDistance = ray.position.y / -ray.direction.y;
    float hitX = ray.direction.x * hitDistance + ray.position.x;
    float hitY = ray.direction.z * hitDistance + ray.position.z;
    level->placementX = (int)floorf(hitX + 0.5f);
    level->placementY = (int)floorf(hitY + 0.5f);
  }

  // the currently hovered/selected tower
  Tower *tower = TowerGetAt(level->placementX, level->placementY);
  // show the range of the tower when hovering/selecting it
  TowerUpdateAllRangeFade(tower, 0.0f);

  BeginMode3D(level->camera);
  DrawLevelGround(level);
  PathFindingMapUpdate(0, 0);
  TowerDrawAll();
  EnemyDraw();
  ProjectileDraw();
  ParticleDraw();
  DrawEnemyPaths(level);

  guiState.isBlocked = 0;

  // Hover rectangle, when the mouse is over the map
  int isHovering = level->placementX >= -5 && level->placementX <= 5 && level->placementY >= -5 && level->placementY <= 5;
  if (isHovering)
  {
    DrawCubeWires((Vector3){level->placementX, 0.75f, level->placementY}, 1.0f, 1.5f, 1.0f, GREEN);
  }

  EndMode3D();

  TowerDrawAllHealthBars(level->camera);

  DrawTitle("Building phase");

  // Draw the context menu when the context menu is active
  if (level->placementContextMenuStatus >= 1)
  {
    float maxHitpoints = 0.0f;
    float hp = 0.0f;
    float damageFactor = 0.0f;
    int32_t sellValue = 0;

    if (tower)
    {
      maxHitpoints = TowerGetMaxHealth(tower);
      hp = maxHitpoints - tower->damage;
      damageFactor = 1.0f - tower->damage / maxHitpoints;
      sellValue = (int32_t) ceilf(TowerTypeGetCosts(tower->towerType) * 0.5f * damageFactor);
    }

    ContextMenuItem menu[12] = {0};
    int menuCount = 0;
    int menuIndex = 0;
    Vector2 anchorLow = GetWorldToScreen((Vector3){level->placementX, -0.25f, level->placementY}, level->camera);
    Vector2 anchorHigh = GetWorldToScreen((Vector3){level->placementX, 2.0f, level->placementY}, level->camera);
    
    if (level->placementContextMenuType == CONTEXT_MENU_TYPE_MAIN)
    {
      DrawLevelBuildingStateMainContextMenu(level, tower, sellValue, hp, maxHitpoints, anchorLow, anchorHigh);
    }
    else if (level->placementContextMenuType == CONTEXT_MENU_TYPE_UPGRADE)
    {
      int totalLevel = tower->upgradeState.damage + tower->upgradeState.speed + tower->upgradeState.range;
      int costs = totalLevel * 4;
      int isMaxLevel = totalLevel >= TOWER_MAX_STAGE;
      menu[menuCount++] = ContextMenuItemText(menuIndex++, TextFormat("%s tower, stage %d%s", 
        TowerTypeGetName(tower->towerType), totalLevel, isMaxLevel ? " (max)" : ""), 0.5f);
      int buttonMenuIndex = menuIndex;
      menu[menuCount++] = ContextMenuItemButton(menuIndex++, TextFormat("Speed: %d (%dG)", tower->upgradeState.speed, costs), 
        OnContextMenuDoUpgrade, (ContextMenuArgs){.tower = tower, .uint8 = UPGRADE_TYPE_SPEED, .int32 = costs});
      menu[menuCount++] = ContextMenuItemButton(menuIndex++, TextFormat("Damage: %d (%dG)", tower->upgradeState.damage, costs),
        OnContextMenuDoUpgrade, (ContextMenuArgs){.tower = tower, .uint8 = UPGRADE_TYPE_DAMAGE, .int32 = costs});
      menu[menuCount++] = ContextMenuItemButton(menuIndex++, TextFormat("Range: %d (%dG)", tower->upgradeState.range, costs),
        OnContextMenuDoUpgrade, (ContextMenuArgs){.tower = tower, .uint8 = UPGRADE_TYPE_RANGE, .int32 = costs});

      // check if buttons should be disabled
      if (isMaxLevel || level->playerGold < costs)
      {
        for (int i = buttonMenuIndex; i < menuCount; i++)
        {
          menu[i].buttonState.isDisabled = 1;
        }
      }

      if (DrawContextMenu(level, anchorLow, anchorHigh, 220, menu))
      {
        level->placementContextMenuType = CONTEXT_MENU_TYPE_MAIN;
      }
    }
    else if (level->placementContextMenuType == CONTEXT_MENU_TYPE_SELL_CONFIRM)
    {
      menu[menuCount++] = ContextMenuItemText(menuIndex++, TextFormat("Sell %s?", TowerTypeGetName(tower->towerType)), 0.5f);
      menu[menuCount++] = ContextMenuItemButton(menuIndex++, "Yes", OnContextMenuSellConfirm, 
        (ContextMenuArgs){.tower = tower, .int32 = sellValue});
      menu[menuCount++] = ContextMenuItemButton(menuIndex++, "No", OnContextMenuSellCancel, (ContextMenuArgs){0});
      Vector2 anchorLow = {GetScreenWidth() * 0.5f, GetScreenHeight() * 0.5f};
      if (DrawContextMenu(level, anchorLow, anchorLow, 150, menu))
      {
        level->placementContextMenuStatus = -1;
        level->placementContextMenuType = CONTEXT_MENU_TYPE_MAIN;
      }
    }
  }
  
  // Activate the context menu when the mouse is clicked and the context menu is not active
  else if (IsMouseButtonReleased(MOUSE_LEFT_BUTTON) && isHovering && level->placementContextMenuStatus <= 0)
  {
    level->placementContextMenuStatus += 1;
  }

  if (level->placementContextMenuStatus == 0)
  {
    if (Button("Reset level", 20, GetScreenHeight() - 40, 160, 30, 0))
    {
      level->nextState = LEVEL_STATE_RESET;
    }
    
    if (Button("Begin waves", GetScreenWidth() - 160, GetScreenHeight() - 40, 160, 30, 0))
    {
      level->nextState = LEVEL_STATE_BATTLE;
    }

  }  
}

void InitBattleStateConditions(Level *level)
{
  level->state = LEVEL_STATE_BATTLE;
  level->nextState = LEVEL_STATE_NONE;
  level->waveEndTimer = 0.0f;
  for (int i = 0; i < 10; i++)
  {
    EnemyWave *wave = &level->waves[i];
    wave->spawned = 0;
    wave->timeToSpawnNext = wave->delay;
  }
}

void DrawLevelBattleState(Level *level)
{
  BeginMode3D(level->camera);
  DrawLevelGround(level);
  TowerUpdateAllRangeFade(0, 0.0f);
  TowerDrawAll();
  EnemyDraw();
  ProjectileDraw();
  ParticleDraw();
  guiState.isBlocked = 0;
  EndMode3D();

  EnemyDrawHealthbars(level->camera);
  TowerDrawAllHealthBars(level->camera);

  if (Button("Reset level", 20, GetScreenHeight() - 40, 160, 30, 0))
  {
    level->nextState = LEVEL_STATE_RESET;
  }

  int maxCount = 0;
  int remainingCount = 0;
  for (int i = 0; i < 10; i++)
  {
    EnemyWave *wave = &level->waves[i];
    if (wave->wave != level->currentWave)
    {
      continue;
    }
    maxCount += wave->count;
    remainingCount += wave->count - wave->spawned;
  }
  int aliveCount = EnemyCount();
  remainingCount += aliveCount;

  const char *text = TextFormat("Battle phase: %03d%%", 100 - remainingCount * 100 / maxCount);
  DrawTitle(text);
}

void DrawLevel(Level *level)
{
  switch (level->state)
  {
    case LEVEL_STATE_BUILDING: DrawLevelBuildingState(level); break;
    case LEVEL_STATE_BUILDING_PLACEMENT: DrawLevelBuildingPlacementState(level); break;
    case LEVEL_STATE_BATTLE: DrawLevelBattleState(level); break;
    case LEVEL_STATE_WON_WAVE: DrawLevelReportWonWave(level); break;
    case LEVEL_STATE_LOST_WAVE: DrawLevelReportLostWave(level); break;
    default: break;
  }

  DrawLevelHud(level);
}

EMSCRIPTEN_KEEPALIVE
void RequestReload()
{
  currentLevel->nextState = LEVEL_STATE_RELOAD;
}

void UpdateLevel(Level *level)
{
  if (level->state == LEVEL_STATE_BATTLE)
  {
    int activeWaves = 0;
    for (int i = 0; i < 10; i++)
    {
      EnemyWave *wave = &level->waves[i];
      if (wave->spawned >= wave->count || wave->wave != level->currentWave)
      {
        continue;
      }
      activeWaves++;
      wave->timeToSpawnNext -= gameTime.deltaTime;
      if (wave->timeToSpawnNext <= 0.0f)
      {
        Enemy *enemy = EnemyTryAdd(wave->enemyType, wave->spawnPosition.x, wave->spawnPosition.y);
        if (enemy)
        {
          wave->timeToSpawnNext = wave->interval;
          wave->spawned++;
        }
      }
    }
    if (TowerGetByType(TOWER_TYPE_BASE) == 0) {
      level->waveEndTimer += gameTime.deltaTime;
      if (level->waveEndTimer >= 2.0f)
      {
        level->nextState = LEVEL_STATE_LOST_WAVE;
      }
    }
    else if (activeWaves == 0 && EnemyCount() == 0)
    {
      level->waveEndTimer += gameTime.deltaTime;
      if (level->waveEndTimer >= 2.0f)
      {
        level->nextState = LEVEL_STATE_WON_WAVE;
      }
    }
  }

  PathFindingMapUpdate(0, 0);
  EnemyUpdate();
  TowerUpdate();
  ProjectileUpdate();
  ParticleUpdate();

  if (level->nextState == LEVEL_STATE_RELOAD)
  {
    if (LoadConfig())
    {
      level->nextState = LEVEL_STATE_RESET;
    }
    else
    {
      level->nextState = level->state;
    }
  }

  if (level->nextState == LEVEL_STATE_RESET)
  {
    currentLevel = loadedLevels[0].initialGold > 0 ? loadedLevels : currentLevel;
    TraceLog(LOG_INFO, "Using level with initialGold = %d", currentLevel->initialGold);
  
    InitLevel(level);
  }
  
  if (level->nextState == LEVEL_STATE_BATTLE)
  {
    InitBattleStateConditions(level);
  }
  
  if (level->nextState == LEVEL_STATE_WON_WAVE)
  {
    level->currentWave++;
    level->state = LEVEL_STATE_WON_WAVE;
  }
  
  if (level->nextState == LEVEL_STATE_LOST_WAVE)
  {
    level->state = LEVEL_STATE_LOST_WAVE;
  }

  if (level->nextState == LEVEL_STATE_BUILDING)
  {
    level->state = LEVEL_STATE_BUILDING;
    level->placementContextMenuStatus = 0;
  }

  if (level->nextState == LEVEL_STATE_BUILDING_PLACEMENT)
  {
    level->state = LEVEL_STATE_BUILDING_PLACEMENT;
    level->placementTransitionPosition = (Vector2){
      level->placementX, level->placementY};
    // initialize the spring to the current position
    level->placementTowerSpring = (PhysicsPoint){
      .position = (Vector3){level->placementX, 8.0f, level->placementY},
      .velocity = (Vector3){0.0f, 0.0f, 0.0f},
    };
    level->placementPhase = PLACEMENT_PHASE_STARTING;
    level->placementTimer = 0.0f;
  }

  if (level->nextState == LEVEL_STATE_WON_LEVEL)
  {
    // make something of this later
    InitLevel(level);
  }

  level->nextState = LEVEL_STATE_NONE;
}

float nextSpawnTime = 0.0f;

int LoadConfig()
{
  char *config = LoadFileText("data/level.txt");
  if (!config)
  {
    TraceLog(LOG_ERROR, "Failed to load level config");
    return 0;
  }

  ParsedGameData gameData = {0};
  if (ParseGameData(&gameData, config))
  {
    for (int i = 0; i < 8; i++)
    {
      EnemyClassConfig *enemyClassConfig = &gameData.enemyClasses[i];
      if (enemyClassConfig->health > 0.0f)
      {
        enemyClassConfigs[i] = *enemyClassConfig;
      }
    }
    
    for (int i = 0; i < 32; i++)
    {
      Level *level = &gameData.levels[i];
      if (level->initialGold > 0)
      {
        loadedLevels[i] = *level;
      }
    }

    for (int i = 0; i < TOWER_TYPE_COUNT; i++)
    {
      TowerTypeConfig *towerTypeConfig = &gameData.towerTypes[i];
      if (towerTypeConfig->maxHealth > 0)
      {
        TowerTypeSetData(i, towerTypeConfig);
      }
    }
  
    currentLevel = loadedLevels[0].initialGold > 0 ? loadedLevels : levels;
  } else {
    TraceLog(LOG_ERROR, "Parsing error: %s", gameData.parseError);
  }

  UnloadFileText(config);

  return gameData.parseError == 0;
}

void InitGame()
{
  TowerInit();
  EnemyInit();
  ProjectileInit();
  ParticleInit();
  PathfindingMapInit(20, 20, (Vector3){-10.0f, 0.0f, -10.0f}, 1.0f);

  InitLevel(currentLevel);
}

//# Immediate GUI functions

void DrawHealthBar(Camera3D camera, Vector3 position, Vector2 screenOffset, float healthRatio, Color barColor, float healthBarWidth)
{
  const float healthBarHeight = 6.0f;
  const float healthBarOffset = 15.0f;
  const float inset = 2.0f;
  const float innerWidth = healthBarWidth - inset * 2;
  const float innerHeight = healthBarHeight - inset * 2;

  Vector2 screenPos = GetWorldToScreen(position, camera);
  screenPos = Vector2Add(screenPos, screenOffset);
  float centerX = screenPos.x - healthBarWidth * 0.5f;
  float topY = screenPos.y - healthBarOffset;
  DrawRectangle(centerX, topY, healthBarWidth, healthBarHeight, BLACK);
  float healthWidth = innerWidth * healthRatio;
  DrawRectangle(centerX + inset, topY + inset, healthWidth, innerHeight, barColor);
}

void DrawBoxedText(const char *text, int x, int y, int width, int height, float alignX, float alignY, Color textColor)
{
  Vector2 textSize = MeasureTextEx(gameFontNormal, text, gameFontNormal.baseSize, 1);
  
  DrawTextEx(gameFontNormal, text, (Vector2){
    x + (width - textSize.x) * alignX, 
    y + (height - textSize.y) * alignY
  }, gameFontNormal.baseSize, 1, textColor);
}

int Button(const char *text, int x, int y, int width, int height, ButtonState *state)
{
  Rectangle bounds = {x, y, width, height};
  int isPressed = 0;
  int isSelected = state && state->isSelected;
  int isDisabled = state && state->isDisabled;
  if (CheckCollisionPointRec(GetMousePosition(), bounds) && !guiState.isBlocked && !isDisabled)
  {
    if (IsMouseButtonPressed(MOUSE_LEFT_BUTTON))
    {
      isPressed = 1;
    }
    guiState.isBlocked = 1;
    DrawTextureNPatch(spriteSheet, isPressed ? uiButtonPressed : uiButtonHovered,
      bounds, Vector2Zero(), 0, WHITE);
  }
  else
  {
    DrawTextureNPatch(spriteSheet, isSelected ? uiButtonHovered : (isDisabled ? uiButtonDisabled : uiButtonNormal),
      bounds, Vector2Zero(), 0, WHITE);
  }
  Vector2 textSize = MeasureTextEx(gameFontNormal, text, gameFontNormal.baseSize, 1);
  Color textColor = isDisabled ? LIGHTGRAY : BLACK;
  DrawTextEx(gameFontNormal, text, (Vector2){x + width / 2 - textSize.x / 2, y + height / 2 - textSize.y / 2}, gameFontNormal.baseSize, 1, textColor);
  return isPressed;
}

//# Main game loop

void GameUpdate()
{
  UpdateLevel(currentLevel);
}

#ifdef PLATFORM_WEB
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
void LogToWeb(int logLevel, const char *text, va_list args)
{
  char logBuffer[1024] = {0};
  vsnprintf(logBuffer, 1024, text, args);
  char escapedBuffer[2048] = {0};
  // escape single quotes
  int outIndex = 0;
  for (int i = 0; i < 1024; i++)
  {
    if (logBuffer[i] == '\'')
    {
      escapedBuffer[outIndex++] = '\\';
    }
    escapedBuffer[outIndex++] = logBuffer[i];
  }
  char js[4096] = {0};
  snprintf(js, 4096, "Module.LogMessage(%d, '%s');", logLevel, escapedBuffer);
  emscripten_run_script(js);
}

void InitWeb()
{
  // create button that adds a textarea with the data/level.txt content
  // together with a button to load the data
  char *js = LoadFileText("data/html-edit.js");
  emscripten_run_script(js);
  UnloadFileText(js);
  TraceLog(LOG_INFO, "Loaded html-edit.js");

  SetTraceLogCallback(LogToWeb);
  TraceLog(LOG_INFO, "JS Logger set");
}
#else
void InitWeb()
{
}
#endif

int main(void)
{
  InitWeb();
  int screenWidth, screenHeight;
  GetPreferredSize(&screenWidth, &screenHeight);
  InitWindow(screenWidth, screenHeight, "Tower defense");
  float gamespeed = 1.0f;
  int frameRate = 30;
  SetTargetFPS(30);

  LoadAssets();
  LoadConfig();
  InitGame();



  float pause = 1.0f;

  while (!WindowShouldClose())
  {
    if (IsPaused()) {
      // canvas is not visible in browser - do nothing
      continue;
    }

    if (IsKeyPressed(KEY_F))
    {
      frameRate = (frameRate + 5) % 30; 
      frameRate = frameRate < 10 ? 10 : frameRate;
      SetTargetFPS(frameRate);
    }

    if (IsKeyPressed(KEY_T))
    {
      gamespeed += 0.1f;
      if (gamespeed > 1.05f) gamespeed = 0.1f;
    }

    if (IsKeyPressed(KEY_P))
    {
      pause = pause > 0.5f ? 0.0f : 1.0f;
    }

    float dt = GetFrameTime() * gamespeed * pause;
    // cap maximum delta time to 0.1 seconds to prevent large time steps
    if (dt > 0.1f) dt = 0.1f;
    gameTime.time += dt;
    gameTime.deltaTime = dt;
    gameTime.frameCount += 1;

    float fixedTimeDiff = gameTime.time - gameTime.fixedTimeStart;
    gameTime.fixedStepCount = (uint8_t)(fixedTimeDiff / gameTime.fixedDeltaTime);

    BeginDrawing();
    ClearBackground((Color){0x4E, 0x63, 0x26, 0xFF});

    GameUpdate();
    DrawLevel(currentLevel);

    if (gamespeed != 1.0f)
      DrawText(TextFormat("Speed: %.1f", gamespeed), GetScreenWidth() - 180, 60, 20, WHITE);
    EndDrawing();

    gameTime.fixedTimeStart += gameTime.fixedStepCount * gameTime.fixedDeltaTime;
  }

  CloseWindow();

  return 0;
}