#include "td_main.h"
#include <raylib.h>
#include <stdio.h>
#include <string.h>

typedef struct ParserState
{
  char *input;
  int position;
  char nextToken[256];
} ParserState;

int ParserStateGetLineNumber(ParserState *state)
{
  int lineNumber = 1;
  for (int i = 0; i < state->position; i++)
  {
    if (state->input[i] == '\n')
    {
      lineNumber++;
    }
  }
  return lineNumber;
}

void ParserStateSkipWhiteSpaces(ParserState *state)
{
  char *input = state->input;
  int pos = state->position;
  int skipped = 1;
  while (skipped)
  {
    skipped = 0;
    if (input[pos] == '-' && input[pos + 1] == '-')
    {
      skipped = 1;
      // skip comments
      while (input[pos] != 0 && input[pos] != '\n')
      {
        pos++;
      }
    }
  
    // skip white spaces and ignore colons
    while (input[pos] != 0 && (input[pos] <= ' ' || input[pos] == ':'))
    {
      skipped = 1;
      pos++;
    }

    // repeat until no more white spaces or comments
  }
  state->position = pos;
}

int ParserStateReadNextToken(ParserState *state)
{
  ParserStateSkipWhiteSpaces(state);

  int i = 0, pos = state->position;
  char *input = state->input;

  // read token
  while (input[pos] != 0 && input[pos] > ' ' && input[pos] != ':' && i < 256)
  {
    state->nextToken[i] = input[pos];
    pos++;
    i++;
  }
  state->position = pos;

  if (i == 0 || i == 256)
  {
    state->nextToken[0] = 0;
    return 0;
  }
  // terminate the token
  state->nextToken[i] = 0;
  return 1;
}

int ParserStateReadNextInt(ParserState *state, int *value)
{
  if (!ParserStateReadNextToken(state))
  {
    return 0;
  }
  // check if the token is a valid integer
  int isSigned = state->nextToken[0] == '-' || state->nextToken[0] == '+';
  for (int i = isSigned; state->nextToken[i] != 0; i++)
  {
    if (state->nextToken[i] < '0' || state->nextToken[i] > '9')
    {
      return 0;
    }
  }
  *value = TextToInteger(state->nextToken);
  return 1;
}

int ParserStateReadNextFloat(ParserState *state, float *value)
{
  if (!ParserStateReadNextToken(state))
  {
    return 0;
  }
  // check if the token is a valid float number
  int isSigned = state->nextToken[0] == '-' || state->nextToken[0] == '+';
  int hasDot = 0;
  for (int i = isSigned; state->nextToken[i] != 0; i++)
  {
    if (state->nextToken[i] == '.')
    {
      if (hasDot)
      {
        return 0;
      }
      hasDot = 1;
    }
    else if (state->nextToken[i] < '0' || state->nextToken[i] > '9')
    {
      return 0;
    }
  }

  *value = TextToFloat(state->nextToken);
  return 1;
}

typedef enum TryReadResult
{
  TryReadResult_NoMatch,
  TryReadResult_Error,
  TryReadResult_Success
} TryReadResult;

TryReadResult ParseGameDataError(ParsedGameData *gameData, ParserState *state, const char *msg)
{
  gameData->parseError = TextFormat("Error at line %d: %s", ParserStateGetLineNumber(state), msg);
  return TryReadResult_Error;
}

TryReadResult ParseGameDataTryReadKeyInt(ParsedGameData *gameData, ParserState *state, const char *key, int *value, int minRange, int maxRange)
{
  if (!TextIsEqual(state->nextToken, key))
  {
    return TryReadResult_NoMatch;
  }

  if (!ParserStateReadNextInt(state, value))
  {
    return ParseGameDataError(gameData, state, TextFormat("Failed to read %s int value", key));
  }

  // range test, if minRange == maxRange, we don't check the range
  if (minRange != maxRange && (*value < minRange || *value > maxRange))
  {
    return ParseGameDataError(gameData, state, TextFormat(
      "Invalid value range for %s, range is [%d, %d], value is %d", 
      key, minRange, maxRange, *value));
  }

  return TryReadResult_Success;
}

TryReadResult ParseGameDataTryReadKeyUInt8(ParsedGameData *gameData, ParserState *state, const char *key, uint8_t *value, uint8_t minRange, uint8_t maxRange)
{
  int intValue = *value;
  TryReadResult result = ParseGameDataTryReadKeyInt(gameData, state, key, &intValue, minRange, maxRange);
  *value = (uint8_t) intValue;
  return result;
}

TryReadResult ParseGameDataTryReadKeyInt16(ParsedGameData *gameData, ParserState *state, const char *key, int16_t *value, int16_t minRange, int16_t maxRange)
{
  int intValue = *value;
  TryReadResult result =ParseGameDataTryReadKeyInt(gameData, state, key, &intValue, minRange, maxRange);
  *value = (int16_t) intValue;
  return result;
}

TryReadResult ParseGameDataTryReadKeyUInt16(ParsedGameData *gameData, ParserState *state, const char *key, uint16_t *value, uint16_t minRange, uint16_t maxRange)
{
  int intValue = *value;
  TryReadResult result =ParseGameDataTryReadKeyInt(gameData, state, key, &intValue, minRange, maxRange);
  *value = (uint16_t) intValue;
  return result;
}

TryReadResult ParseGameDataTryReadKeyIntVec2(ParsedGameData *gameData, ParserState *state, const char *key, 
  Vector2 *vector, Vector2 minRange, Vector2 maxRange)
{
  if (!TextIsEqual(state->nextToken, key))
  {
    return TryReadResult_NoMatch;
  }

  ParserState start = *state;
  int x = 0, y = 0;
  int minXRange = (int)minRange.x, maxXRange = (int)maxRange.x;
  int minYRange = (int)minRange.y, maxYRange = (int)maxRange.y;

  if (!ParserStateReadNextInt(state, &x))
  {
    // use start position to report the error for this KEY
    return ParseGameDataError(gameData, &start, TextFormat("Failed to read %s x int value", key));
  }

  // range test, if minRange == maxRange, we don't check the range
  if (minXRange != maxXRange && (x < minXRange || x > maxXRange))
  {
    // use current position to report the error for x value
    return ParseGameDataError(gameData, state, TextFormat(
      "Invalid value x range for %s, range is [%d, %d], value is %d", 
      key, minXRange, maxXRange, x));
  }

  if (!ParserStateReadNextInt(state, &y))
  {
    // use start position to report the error for this KEY
    return ParseGameDataError(gameData, &start, TextFormat("Failed to read %s y int value", key));
  }

  if (minYRange != maxYRange && (y < minYRange || y > maxYRange))
  {
    // use current position to report the error for y value
    return ParseGameDataError(gameData, state, TextFormat(
      "Invalid value y range for %s, range is [%d, %d], value is %d", 
      key, minYRange, maxYRange, y));
  }

  vector->x = (float)x;
  vector->y = (float)y;

  return TryReadResult_Success;
}

TryReadResult ParseGameDataTryReadKeyFloat(ParsedGameData *gameData, ParserState *state, const char *key, float *value, float minRange, float maxRange)
{
  if (!TextIsEqual(state->nextToken, key))
  {
    return TryReadResult_NoMatch;
  }

  if (!ParserStateReadNextFloat(state, value))
  {
    return ParseGameDataError(gameData, state, TextFormat("Failed to read %s float value", key));
  }

  // range test, if minRange == maxRange, we don't check the range
  if (minRange != maxRange && (*value < minRange || *value > maxRange))
  {
    return ParseGameDataError(gameData, state, TextFormat(
      "Invalid value range for %s, range is [%f, %f], value is %f", 
      key, minRange, maxRange, *value));
  }

  return TryReadResult_Success;
}

// The enumNames is a null-terminated array of strings that represent the enum values
TryReadResult ParseGameDataTryReadEnum(ParsedGameData *gameData, ParserState *state, const char *key, int *value, const char *enumNames[], int *enumValues)
{
  if (!TextIsEqual(state->nextToken, key))
  {
    return TryReadResult_NoMatch;
  }

  if (!ParserStateReadNextToken(state))
  {
    return ParseGameDataError(gameData, state, TextFormat("Failed to read %s enum value", key));
  }

  for (int i = 0; enumNames[i] != 0; i++)
  {
    if (TextIsEqual(state->nextToken, enumNames[i]))
    {
      *value = enumValues[i];
      return TryReadResult_Success;
    }
  }

  return ParseGameDataError(gameData, state, TextFormat("Invalid value for %s", key));
}

TryReadResult ParseGameDataTryReadKeyEnemyTypeId(ParsedGameData *gameData, ParserState *state, const char *key, uint8_t *enemyTypeId, uint8_t minRange, uint8_t maxRange)
{
  int enemyClassId = *enemyTypeId;
  TryReadResult result = ParseGameDataTryReadEnum(gameData, state, key, (int *)&enemyClassId, 
      (const char *[]){"ENEMY_TYPE_MINION", "ENEMY_TYPE_RUNNER", "ENEMY_TYPE_SHIELD", "ENEMY_TYPE_BOSS", 0}, 
      (int[]){ENEMY_TYPE_MINION, ENEMY_TYPE_RUNNER, ENEMY_TYPE_SHIELD, ENEMY_TYPE_BOSS});
  if (minRange != maxRange)
  {
    enemyClassId = enemyClassId < minRange ? minRange : enemyClassId;
    enemyClassId = enemyClassId > maxRange ? maxRange : enemyClassId;
  }
  *enemyTypeId = (uint8_t) enemyClassId;
  return result;
}

TryReadResult ParseGameDataTryReadKeyTowerTypeId(ParsedGameData *gameData, ParserState *state, const char *key, uint8_t *towerTypeId, uint8_t minRange, uint8_t maxRange)
{
  int towerType = *towerTypeId;
  TryReadResult result = ParseGameDataTryReadEnum(gameData, state, key, &towerType, 
      (const char *[]){"TOWER_TYPE_BASE", "TOWER_TYPE_ARCHER", "TOWER_TYPE_BALLISTA", "TOWER_TYPE_CATAPULT", "TOWER_TYPE_WALL", 0}, 
      (int[]){TOWER_TYPE_BASE, TOWER_TYPE_ARCHER, TOWER_TYPE_BALLISTA, TOWER_TYPE_CATAPULT, TOWER_TYPE_WALL});
  if (minRange != maxRange)
  {
    towerType = towerType < minRange ? minRange : towerType;
    towerType = towerType > maxRange ? maxRange : towerType;
  }
  *towerTypeId = (uint8_t) towerType;
  return result;
}

TryReadResult ParseGameDataTryReadKeyProjectileTypeId(ParsedGameData *gameData, ParserState *state, const char *key, uint8_t *towerTypeId, uint8_t minRange, uint8_t maxRange)
{
  int towerType = *towerTypeId;
  TryReadResult result = ParseGameDataTryReadEnum(gameData, state, key, &towerType, 
      (const char *[]){"PROJECTILE_TYPE_ARROW", "PROJECTILE_TYPE_BALLISTA", "PROJECTILE_TYPE_CATAPULT", 0}, 
      (int[]){PROJECTILE_TYPE_ARROW, PROJECTILE_TYPE_BALLISTA, PROJECTILE_TYPE_CATAPULT});
  if (minRange != maxRange)
  {
    towerType = towerType < minRange ? minRange : towerType;
    towerType = towerType > maxRange ? maxRange : towerType;
  }
  *towerTypeId = (uint8_t) towerType;
  return result;
}


//----------------------------------------------------------------
//# Defines for compact struct field parsing
// A FIELDS(GENERATEr) is to be defined that will be called for each field of the struct
// See implementations below for how this is used
#define GENERATE_READFIELD_SWITCH(owner, name, type, min, max)\
  switch (ParseGameDataTryReadKey##type(gameData, state, #name, &owner->name, min, max))\
  {\
    case TryReadResult_NoMatch: break;\
    case TryReadResult_Success:\
      if (name##Initialized) {\
        return ParseGameDataError(gameData, state, #name " already initialized");\
      }\
      name##Initialized = 1;\
      continue;\
    case TryReadResult_Error: return TryReadResult_Error;\
  }
#define GENERATE_READFIELD_SWITCH_OPTIONAL(owner, name, type, def, min, max)\
  GENERATE_READFIELD_SWITCH(owner, name, type, min, max)
#define GENERATE_FIELD_INIT_DECLARATIONS(owner, name, type, min, max) int name##Initialized = 0;
#define GENERATE_FIELD_INIT_DECLARATIONS_OPTIONAL(owner, name, type, def, min, max) int name##Initialized = 0; owner->name = def;
#define GENERATE_FIELD_INIT_CHECK(owner, name, type, min, max) \
  if (!name##Initialized) { \
    return ParseGameDataError(gameData, state, #name " not initialized"); \
  }
#define GENERATE_FIELD_INIT_CHECK_OPTIONAL(owner, name, type, def, min, max)

#define GENERATE_FIELD_PARSING \
  FIELDS(GENERATE_FIELD_INIT_DECLARATIONS, GENERATE_FIELD_INIT_DECLARATIONS_OPTIONAL)\
  while (1)\
  {\
    ParserState prevState = *state;\
    \
    if (!ParserStateReadNextToken(state))\
    {\
      /* end of file */\
      break;\
    }\
    FIELDS(GENERATE_READFIELD_SWITCH, GENERATE_READFIELD_SWITCH_OPTIONAL)\
    /* no match, return to previous state and break */\
    *state = prevState;\
    break;\
  } \
  FIELDS(GENERATE_FIELD_INIT_CHECK, GENERATE_FIELD_INIT_CHECK_OPTIONAL)\

// END OF DEFINES

TryReadResult ParseGameDataTryReadWaveSection(ParsedGameData *gameData, ParserState *state)
{
  if (!TextIsEqual(state->nextToken, "Wave"))
  {
    return TryReadResult_NoMatch;
  }

  Level *level = &gameData->levels[gameData->lastLevelIndex];
  EnemyWave *wave = 0;
  for (int i = 0; i < ENEMY_MAX_WAVE_COUNT; i++)
  {
    if (level->waves[i].count == 0)
    {
      wave = &level->waves[i];
      break;
    }
  }

  if (wave == 0)
  {
    return ParseGameDataError(gameData, state, 
      TextFormat("Wave section overflow (max is %d)", ENEMY_MAX_WAVE_COUNT));
  }

#define FIELDS(MANDATORY, OPTIONAL) \
  MANDATORY(wave, wave, UInt8, 0, ENEMY_MAX_WAVE_COUNT - 1) \
  MANDATORY(wave, count, UInt16, 1, 1000) \
  MANDATORY(wave, delay, Float, 0.0f, 1000.0f) \
  MANDATORY(wave, interval, Float, 0.0f, 1000.0f) \
  MANDATORY(wave, spawnPosition, IntVec2, ((Vector2){-10.0f, -10.0f}), ((Vector2){10.0f, 10.0f})) \
  MANDATORY(wave, enemyType, EnemyTypeId, 0, 0)
  
  GENERATE_FIELD_PARSING
#undef FIELDS

  return TryReadResult_Success;
}

TryReadResult ParseGameDataTryReadEnemyClassSection(ParsedGameData *gameData, ParserState *state)
{
  uint8_t enemyClassId;
  
  switch (ParseGameDataTryReadKeyEnemyTypeId(gameData, state, "EnemyClass", &enemyClassId, 0, 7))
  {
    case TryReadResult_Success: break;
    case TryReadResult_NoMatch: return TryReadResult_NoMatch;
    case TryReadResult_Error: return TryReadResult_Error;
  }

  EnemyClassConfig *enemyClass = &gameData->enemyClasses[enemyClassId];

#define FIELDS(MANDATORY, OPTIONAL) \
  MANDATORY(enemyClass, speed, Float, 0.1f, 1000.0f) \
  MANDATORY(enemyClass, health, Float, 1, 1000000) \
  MANDATORY(enemyClass, radius, Float, 0.0f, 10.0f) \
  MANDATORY(enemyClass, requiredContactTime, Float, 0.0f, 1000.0f) \
  MANDATORY(enemyClass, maxAcceleration, Float, 0.0f, 1000.0f) \
  MANDATORY(enemyClass, explosionDamage, Float, 0.0f, 1000.0f) \
  MANDATORY(enemyClass, explosionRange, Float, 0.0f, 1000.0f) \
  MANDATORY(enemyClass, explosionPushbackPower, Float, 0.0f, 1000.0f) \
  MANDATORY(enemyClass, goldValue, Int, 1, 1000000)

  GENERATE_FIELD_PARSING
#undef FIELDS

  return TryReadResult_Success;
}

TryReadResult ParseGameDataTryReadTowerTypeConfigSection(ParsedGameData *gameData, ParserState *state)
{
  uint8_t towerTypeId;
  
  switch (ParseGameDataTryReadKeyTowerTypeId(gameData, state, "TowerTypeConfig", &towerTypeId, 0, TOWER_TYPE_COUNT - 1))
  {
    case TryReadResult_Success: break;
    case TryReadResult_NoMatch: return TryReadResult_NoMatch;
    case TryReadResult_Error: return TryReadResult_Error;
  }

  TowerTypeConfig *towerType = &gameData->towerTypes[towerTypeId];
  HitEffectConfig *hitEffect = &towerType->hitEffect;

#define FIELDS(MANDATORY, OPTIONAL) \
  MANDATORY(towerType, maxHealth, UInt16, 0, 0) \
  OPTIONAL(towerType, cooldown, Float, 0, 0.0f, 1000.0f) \
  OPTIONAL(towerType, maxUpgradeCooldown, Float, 0, 0.0f, 1000.0f) \
  OPTIONAL(towerType, range, Float, 0, 0.0f, 50.0f) \
  OPTIONAL(towerType, maxUpgradeRange, Float, 0, 0.0f, 50.0f) \
  OPTIONAL(towerType, projectileSpeed, Float, 0, 0.0f, 100.0f) \
  OPTIONAL(towerType, cost, UInt8, 0, 0, 255) \
  OPTIONAL(towerType, projectileType, ProjectileTypeId, 0, 0, 32)\
  OPTIONAL(hitEffect, damage, Float, 0, 0, 100000.0f) \
  OPTIONAL(hitEffect, maxUpgradeDamage, Float, 0, 0, 100000.0f) \
  OPTIONAL(hitEffect, areaDamageRadius, Float, 0, 0, 100000.0f) \
  OPTIONAL(hitEffect, pushbackPowerDistance, Float, 0, 0, 100000.0f)

  GENERATE_FIELD_PARSING
#undef FIELDS

  return TryReadResult_Success;
}

TryReadResult ParseGameDataTryReadLevelSection(ParsedGameData *gameData, ParserState *state)
{
  int levelId;
  TryReadResult result = ParseGameDataTryReadKeyInt(gameData, state, "Level", &levelId, 0, 31);
  if (result != TryReadResult_Success)
  {
    return result;
  }
  
  gameData->lastLevelIndex = levelId;
  Level *level = &gameData->levels[levelId];

  // since we require the initialGold to be initialized with at least 1, we can use it as a flag
  // to detect if the level was already initialized
  if (level->initialGold != 0)
  {
    return ParseGameDataError(gameData, state, TextFormat("level %d already initialized", levelId));
  }

  int initialGoldInitialized = 0;

  while (1)
  {
    // try to read the next token and if we don't know how to GENERATE it,
    // we rewind and return
    ParserState prevState = *state;
    
    if (!ParserStateReadNextToken(state))
    {
      // end of file
      break;
    }

    switch (ParseGameDataTryReadKeyInt(gameData, state, "initialGold", &level->initialGold, 1, 100000))
    {
      case TryReadResult_Success: 
        if (initialGoldInitialized)
        {
          return ParseGameDataError(gameData, state, "initialGold already initialized");
        }
        initialGoldInitialized = 1;
        continue;
      case TryReadResult_Error: return TryReadResult_Error;
      case TryReadResult_NoMatch: break;
    }

    switch (ParseGameDataTryReadWaveSection(gameData, state))
    {
      case TryReadResult_Success: continue;
      case TryReadResult_NoMatch: break;
      case TryReadResult_Error: return TryReadResult_Error;
    }

    // no match, return to previous state and break
    *state = prevState;
    break;
  }

  if (!initialGoldInitialized)
  {
    return ParseGameDataError(gameData, state, "initialGold not initialized");
  }

  return TryReadResult_Success;
}

int ParseGameData(ParsedGameData *gameData, const char *input)
{
  ParserState state = (ParserState){(char *)input, 0, {0}};
  *gameData = (ParsedGameData){0};
  gameData->lastLevelIndex = -1;

  while (ParserStateReadNextToken(&state))
  {
    switch (ParseGameDataTryReadLevelSection(gameData, &state))
    {
      case TryReadResult_Success: continue;
      case TryReadResult_Error: return 0;
      case TryReadResult_NoMatch: break;
    }

    switch (ParseGameDataTryReadEnemyClassSection(gameData, &state))
    {
      case TryReadResult_Success: continue;
      case TryReadResult_Error: return 0;
      case TryReadResult_NoMatch: break;
    }
    
    switch (ParseGameDataTryReadTowerTypeConfigSection(gameData, &state))
    {
      case TryReadResult_Success: continue;
      case TryReadResult_Error: return 0;
      case TryReadResult_NoMatch: break;
    }

    // any other token is considered an error
    ParseGameDataError(gameData, &state, TextFormat("Unexpected token: %s", state.nextToken));
    return 0;
  }

  return 1;
}

#define EXPECT(cond, msg) if (!(cond)) { printf("Error %s:%d: %s\n", __FILE__, __LINE__, msg); failedCount++; } else { passedCount++; }

int RunParseTests()
{
  int passedCount = 0, failedCount = 0;
  ParsedGameData gameData;
  const char *input;

  input ="Level 7\n  initialGold 100\nLevel 2 initialGold 200";
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 1, "Failed to parse level section");
  EXPECT(gameData.lastLevelIndex == 2, "lastLevelIndex should be 2");
  EXPECT(gameData.levels[7].initialGold == 100, "initialGold should be 100");
  EXPECT(gameData.levels[2].initialGold == 200, "initialGold should be 200");

  input ="Level 392\n";
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 0, "Expected to fail parsing level section");
  EXPECT(TextFindIndex(gameData.parseError, "Invalid value range for Level") >= 0, "Expected invalid value range error");
  EXPECT(TextFindIndex(gameData.parseError, "line 1") >= 0, "Expected to find line number 1");
  
  input ="Level 3\n initialGold -34";
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 0, "Expected to fail parsing level section");
  EXPECT(TextFindIndex(gameData.parseError, "Invalid value range for initialGold") >= 0, "Expected invalid value range error");
  EXPECT(TextFindIndex(gameData.parseError, "line 2") >= 0, "Expected to find line number 1");
  
  input ="Level 3\n initialGold 2\n initialGold 3";
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 0, "Expected to fail parsing level section");
  EXPECT(TextFindIndex(gameData.parseError, "line 3") >= 0, "Expected to find line number 1");

  input ="Level 3";
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 0, "Expected to fail parsing level section");

  input ="Level 7\n  initialGold 100\nLevel 7 initialGold 200";
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 0, "Expected to fail parsing level section");
  EXPECT(TextFindIndex(gameData.parseError, "line 3") >= 0, "Expected to find line number 1");

  input =
    "Level 7\n  initialGold 100\n"
    "Wave\n"
    "count 1 wave 2\n"
    "interval 0.5\n"
    "delay 1.0\n"
    "spawnPosition -3 4\n"
    "enemyType: ENEMY_TYPE_SHIELD"
    ;
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 1, "Expected to succeed parsing level/wave section");
  EXPECT(gameData.levels[7].waves[0].count == 1, "Expected wave count to be 1");
  EXPECT(gameData.levels[7].waves[0].wave == 2, "Expected wave to be 2");
  EXPECT(gameData.levels[7].waves[0].interval == 0.5f, "Expected interval to be 0.5");
  EXPECT(gameData.levels[7].waves[0].delay == 1.0f, "Expected delay to be 1.0");
  EXPECT(gameData.levels[7].waves[0].spawnPosition.x == -3, "Expected spawnPosition.x to be 3");
  EXPECT(gameData.levels[7].waves[0].spawnPosition.y == 4, "Expected spawnPosition.y to be 4");
  EXPECT(gameData.levels[7].waves[0].enemyType == ENEMY_TYPE_SHIELD, "Expected enemyType to be ENEMY_TYPE_SHIELD");
  
  // for every entry in the wave section, we want to verify that if that value is 
  // missing, the parser will produce an error. We can do that by commenting out each
  // line individually in a loop - just replacing the two leading spaces with two dashes
  const char *testString =
    "Level 7 initialGold 100\n"
    "Wave\n"
    "  count 1\n"
    "  wave 2\n"
    "  interval 0.5\n"
    "  delay 1.0\n"
    "  spawnPosition 3 -4\n"
    "  enemyType: ENEMY_TYPE_SHIELD";
  for (int i = 0; testString[i]; i++)
  {
    if (testString[i] == '\n' && testString[i + 1] == ' ' && testString[i + 2] == ' ')
    {
      char copy[1024];
      strcpy(copy, testString);
      // commentify!
      copy[i + 1] = '-';
      copy[i + 2] = '-';
      gameData = (ParsedGameData) {0};
      EXPECT(ParseGameData(&gameData, copy) == 0, "Expected to fail parsing level/wave section");
    }
  }

  // test wave section missing data / incorrect data

  input =
    "Level 7\n  initialGold 100\n"
    "Wave\n"
    "count 1 wave 2\n"
    "interval 0.5\n"
    "delay 1.0\n"
    "spawnPosition -3\n" // missing y
    "enemyType: ENEMY_TYPE_SHIELD"
    ;
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 0, "Expected to fail parsing level/wave section");
  EXPECT(TextFindIndex(gameData.parseError, "line 7") >= 0, "Expected to find line 7");
  
  input =
    "Level 7\n  initialGold 100\n"
    "Wave\n"
    "count 1.0 wave 2\n"
    "interval 0.5\n"
    "delay 1.0\n"
    "spawnPosition -3\n" // missing y
    "enemyType: ENEMY_TYPE_SHIELD"
    ;
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 0, "Expected to fail parsing level/wave section");
  EXPECT(TextFindIndex(gameData.parseError, "line 4") >= 0, "Expected to find line 3");

  // enemy class config parsing tests
  input =
    "EnemyClass ENEMY_TYPE_MINION\n"
    "  health: 10.0\n"
    "  speed: 0.6\n"
    "  radius: 0.25\n"
    "  maxAcceleration: 1.0\n"
    "  explosionDamage: 1.0\n"
    "  requiredContactTime: 0.5\n"
    "  explosionRange: 1.0\n"
    "  explosionPushbackPower: 0.25\n"
    "  goldValue: 1\n"
    ;
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 1, "Expected to succeed parsing enemy class section");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].health == 10.0f, "Expected health to be 10.0");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].speed == 0.6f, "Expected speed to be 0.6");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].radius == 0.25f, "Expected radius to be 0.25");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].maxAcceleration == 1.0f, "Expected maxAcceleration to be 1.0");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].explosionDamage == 1.0f, "Expected explosionDamage to be 1.0");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].requiredContactTime == 0.5f, "Expected requiredContactTime to be 0.5");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].explosionRange == 1.0f, "Expected explosionRange to be 1.0");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].explosionPushbackPower == 0.25f, "Expected explosionPushbackPower to be 0.25");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].goldValue == 1, "Expected goldValue to be 1");

  testString =
    "EnemyClass ENEMY_TYPE_MINION\n"
    "  health: 10.0\n"
    "  speed: 0.6\n"
    "  radius: 0.25\n"
    "  maxAcceleration: 1.0\n"
    "  explosionDamage: 1.0\n"
    "  requiredContactTime: 0.5\n"
    "  explosionRange: 1.0\n"
    "  explosionPushbackPower: 0.25\n"
    "  goldValue: 1\n";
  for (int i = 0; testString[i]; i++)
  {
    if (testString[i] == '\n' && testString[i + 1] == ' ' && testString[i + 2] == ' ')
    {
      char copy[1024];
      strcpy(copy, testString);
      // commentify!
      copy[i + 1] = '-';
      copy[i + 2] = '-';
      gameData = (ParsedGameData) {0};
      EXPECT(ParseGameData(&gameData, copy) == 0, "Expected to fail parsing EnemyClass section");
    }
  }

  input =
    "TowerTypeConfig TOWER_TYPE_ARCHER\n"
    "  cooldown: 0.5\n"
    "  maxUpgradeCooldown: 0.25\n"
    "  range: 3\n"
    "  maxUpgradeRange: 5\n"
    "  projectileSpeed: 4.0\n"
    "  cost: 5\n"
    "  maxHealth: 10\n"
    "  projectileType: PROJECTILE_TYPE_ARROW\n" 
    "  damage: 0.5\n"
    "  maxUpgradeDamage: 1.5\n"
    "  areaDamageRadius: 0\n"
    "  pushbackPowerDistance: 0\n"
    ;
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 1, "Expected to succeed parsing tower type section");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].cooldown == 0.5f, "Expected cooldown to be 0.5");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].maxUpgradeCooldown == 0.25f, "Expected maxUpgradeCooldown to be 0.25");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].range == 3.0f, "Expected range to be 3.0");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].maxUpgradeRange == 5.0f, "Expected maxUpgradeRange to be 5.0");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].projectileSpeed == 4.0f, "Expected projectileSpeed to be 4.0");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].cost == 5, "Expected cost to be 5");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].maxHealth == 10, "Expected maxHealth to be 10");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].projectileType == PROJECTILE_TYPE_ARROW, "Expected projectileType to be PROJECTILE_TYPE_ARROW");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].hitEffect.damage == 0.5f, "Expected damage to be 0.5");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].hitEffect.maxUpgradeDamage == 1.5f, "Expected maxUpgradeDamage to be 1.5");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].hitEffect.areaDamageRadius == 0.0f, "Expected areaDamageRadius to be 0.0");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].hitEffect.pushbackPowerDistance == 0.0f, "Expected pushbackPowerDistance to be 0.0");

  input =
    "TowerTypeConfig TOWER_TYPE_ARCHER\n"
    "  maxHealth: 10\n"
    "  cooldown: 0.5\n"
    ;
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 1, "Expected to succeed parsing tower type section");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].cooldown == 0.5f, "Expected cooldown to be 0.5");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].maxHealth == 10, "Expected maxHealth to be 10");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].cost == 0, "Expected cost to be 0");

  
  input =
    "TowerTypeConfig TOWER_TYPE_ARCHER\n"
    "  cooldown: 0.5\n"
    ;
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 0, "Expected to fail parsing tower type section");
  EXPECT(TextFindIndex(gameData.parseError, "maxHealth not initialized") >= 0, "Expected to find maxHealth not initialized");

  input =
    "TowerTypeConfig TOWER_TYPE_ARCHER\n"
    "  maxHealth: 10\n"
    "  foobar: 0.5\n"
    ;
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, input) == 0, "Expected to fail parsing tower type section");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].maxHealth == 10, "Expected maxHealth to be 10");
  EXPECT(gameData.towerTypes[TOWER_TYPE_ARCHER].cost == 0, "Expected cost to be 0");

  printf("Passed %d test(s), Failed %d\n", passedCount, failedCount);

  return failedCount;
}
