#include "td_main.h"
#include <raylib.h>
#include <stdio.h>
#include <string.h>

typedef struct ParserState
{
  char *input;
  int position;
  char nextToken[256];
} ParserState;

int ParserStateGetLineNumber(ParserState *state)
{
  int lineNumber = 1;
  for (int i = 0; i < state->position; i++)
  {
    if (state->input[i] == '\n')
    {
      lineNumber++;
    }
  }
  return lineNumber;
}

void ParserStateSkipWhiteSpaces(ParserState *state)
{
  char *input = state->input;
  int pos = state->position;
  int skipped = 1;
  while (skipped)
  {
    skipped = 0;
    if (input[pos] == '-' && input[pos + 1] == '-')
    {
      skipped = 1;
      // skip comments
      while (input[pos] != 0 && input[pos] != '\n')
      {
        pos++;
      }
    }
  
    // skip white spaces and ignore colons
    while (input[pos] != 0 && (input[pos] <= ' ' || input[pos] == ':'))
    {
      skipped = 1;
      pos++;
    }

    // repeat until no more white spaces or comments
  }
  state->position = pos;
}

int ParserStateReadNextToken(ParserState *state)
{
  ParserStateSkipWhiteSpaces(state);

  int i = 0, pos = state->position;
  char *input = state->input;

  // read token
  while (input[pos] != 0 && input[pos] > ' ' && input[pos] != ':' && i < 256)
  {
    state->nextToken[i] = input[pos];
    pos++;
    i++;
  }
  state->position = pos;

  if (i == 0 || i == 256)
  {
    state->nextToken[0] = 0;
    return 0;
  }
  // terminate the token
  state->nextToken[i] = 0;
  return 1;
}

int ParserStateReadNextInt(ParserState *state, int *value)
{
  if (!ParserStateReadNextToken(state))
  {
    return 0;
  }
  // check if the token is a valid integer
  int isSigned = state->nextToken[0] == '-' || state->nextToken[0] == '+';
  for (int i = isSigned; state->nextToken[i] != 0; i++)
  {
    if (state->nextToken[i] < '0' || state->nextToken[i] > '9')
    {
      return 0;
    }
  }
  *value = TextToInteger(state->nextToken);
  return 1;
}

int ParserStateReadNextFloat(ParserState *state, float *value)
{
  if (!ParserStateReadNextToken(state))
  {
    return 0;
  }
  // check if the token is a valid float number
  int isSigned = state->nextToken[0] == '-' || state->nextToken[0] == '+';
  int hasDot = 0;
  for (int i = isSigned; state->nextToken[i] != 0; i++)
  {
    if (state->nextToken[i] == '.')
    {
      if (hasDot)
      {
        return 0;
      }
      hasDot = 1;
    }
    else if (state->nextToken[i] < '0' || state->nextToken[i] > '9')
    {
      return 0;
    }
  }

  *value = TextToFloat(state->nextToken);
  return 1;
}

typedef struct ParsedGameData
{
  const char *parseError;
  Level levels[32];
  int lastLevelIndex;
  TowerTypeConfig towerTypes[TOWER_TYPE_COUNT];
  EnemyClassConfig enemyClasses[8];
} ParsedGameData;

typedef enum TryReadResult
{
  TryReadResult_NoMatch,
  TryReadResult_Error,
  TryReadResult_Success
} TryReadResult;

TryReadResult ParseGameDataError(ParsedGameData *gameData, ParserState *state, const char *msg)
{
  gameData->parseError = TextFormat("Error at line %d: %s", ParserStateGetLineNumber(state), msg);
  return TryReadResult_Error;
}

TryReadResult ParseGameDataTryReadKeyInt(ParsedGameData *gameData, ParserState *state, const char *key, int *value, int minRange, int maxRange)
{
  if (!TextIsEqual(state->nextToken, key))
  {
    return TryReadResult_NoMatch;
  }

  if (!ParserStateReadNextInt(state, value))
  {
    return ParseGameDataError(gameData, state, TextFormat("Failed to read %s int value", key));
  }

  // range test, if minRange == maxRange, we don't check the range
  if (minRange != maxRange && (*value < minRange || *value > maxRange))
  {
    return ParseGameDataError(gameData, state, TextFormat(
      "Invalid value range for %s, range is [%d, %d], value is %d", 
      key, minRange, maxRange, *value));
  }

  return TryReadResult_Success;
}

TryReadResult ParseGameDataTryReadKeyIntVec2(ParsedGameData *gameData, ParserState *state, const char *key, 
  int *x, int minXRange, int maxXRange, int *y, int minYRange, int maxYRange)
{
  if (!TextIsEqual(state->nextToken, key))
  {
    return TryReadResult_NoMatch;
  }

  ParserState start = *state;

  if (!ParserStateReadNextInt(state, x))
  {
    // use start position to report the error for this KEY
    return ParseGameDataError(gameData, &start, TextFormat("Failed to read %s x int value", key));
  }

  // range test, if minRange == maxRange, we don't check the range
  if (minXRange != maxXRange && (*x < minXRange || *x > maxXRange))
  {
    // use current position to report the error for x value
    return ParseGameDataError(gameData, state, TextFormat(
      "Invalid value x range for %s, range is [%d, %d], value is %d", 
      key, minXRange, maxXRange, *x));
  }

  if (!ParserStateReadNextInt(state, y))
  {
    // use start position to report the error for this KEY
    return ParseGameDataError(gameData, &start, TextFormat("Failed to read %s y int value", key));
  }

  if (minYRange != maxYRange && (*y < minYRange || *y > maxYRange))
  {
    // use current position to report the error for y value
    return ParseGameDataError(gameData, state, TextFormat(
      "Invalid value y range for %s, range is [%d, %d], value is %d", 
      key, minYRange, maxYRange, *y));
  }

  return TryReadResult_Success;
}

TryReadResult ParseGameDataTryReadKeyFloat(ParsedGameData *gameData, ParserState *state, const char *key, float *value, float minRange, float maxRange)
{
  if (!TextIsEqual(state->nextToken, key))
  {
    return TryReadResult_NoMatch;
  }

  if (!ParserStateReadNextFloat(state, value))
  {
    return ParseGameDataError(gameData, state, TextFormat("Failed to read %s float value", key));
  }

  // range test, if minRange == maxRange, we don't check the range
  if (minRange != maxRange && (*value < minRange || *value > maxRange))
  {
    return ParseGameDataError(gameData, state, TextFormat(
      "Invalid value range for %s, range is [%f, %f], value is %f", 
      key, minRange, maxRange, *value));
  }

  return TryReadResult_Success;
}

// The enumNames is a null-terminated array of strings that represent the enum values
TryReadResult ParseGameDataTryReadEnum(ParsedGameData *gameData, ParserState *state, const char *key, int *value, const char *enumNames[], int *enumValues)
{
  if (!TextIsEqual(state->nextToken, key))
  {
    return TryReadResult_NoMatch;
  }

  if (!ParserStateReadNextToken(state))
  {
    return ParseGameDataError(gameData, state, TextFormat("Failed to read %s enum value", key));
  }

  for (int i = 0; enumNames[i] != 0; i++)
  {
    if (TextIsEqual(state->nextToken, enumNames[i]))
    {
      *value = enumValues[i];
      return TryReadResult_Success;
    }
  }

  return ParseGameDataError(gameData, state, TextFormat("Invalid value for %s", key));
}

TryReadResult ParseGameDataTryReadEnemyTypeId(ParsedGameData *gameData, ParserState *state, const char *key, uint8_t *enemyTypeId)
{
  int enemyClassId;
  switch (ParseGameDataTryReadEnum(gameData, state, key, (int *)&enemyClassId, 
      (const char *[]){"ENEMY_TYPE_MINION", "ENEMY_TYPE_RUNNER", "ENEMY_TYPE_SHIELD", "ENEMY_TYPE_BOSS", 0}, 
      (int[]){ENEMY_TYPE_MINION, ENEMY_TYPE_RUNNER, ENEMY_TYPE_SHIELD, ENEMY_TYPE_BOSS}))
  {
    case TryReadResult_Success: 
      *enemyTypeId = (uint8_t) enemyClassId;
      return TryReadResult_Success;
    case TryReadResult_Error: return TryReadResult_Error;
  }
  return TryReadResult_NoMatch;
}

TryReadResult ParseGameDataTryReadWaveSection(ParsedGameData *gameData, ParserState *state)
{
  if (!TextIsEqual(state->nextToken, "Wave"))
  {
    return TryReadResult_NoMatch;
  }

  Level *level = &gameData->levels[gameData->lastLevelIndex];
  EnemyWave *wave = 0;
  for (int i = 0; i < ENEMY_MAX_WAVE_COUNT; i++)
  {
    if (level->waves[i].count == 0)
    {
      wave = &level->waves[i];
      break;
    }
  }

  if (wave == 0)
  {
    return ParseGameDataError(gameData, state, 
      TextFormat("Wave section overflow (max is %d)", ENEMY_MAX_WAVE_COUNT));
  }

  int waveInitialized = 0;
  int countInitialized = 0;
  int delayInitialized = 0;
  int intervalInitialized = 0;
  int spawnPositionInitialized = 0;
  int enemyTypeInitialized = 0;

#define ASSURE_IS_NOT_INITIALIZED(name) if (name##Initialized) { \
  return ParseGameDataError(gameData, state, #name " already initialized"); } \
  name##Initialized = 1

  while (1)
  {
    ParserState prevState = *state;
    
    if (!ParserStateReadNextToken(state))
    {
      // end of file
      break;
    }

    int value;
    switch (ParseGameDataTryReadKeyInt(gameData, state, "wave", &value, 0, ENEMY_MAX_WAVE_COUNT - 1))
    {
      case TryReadResult_Success:
        ASSURE_IS_NOT_INITIALIZED(wave);
        wave->wave = (uint8_t) value;
        continue;
      case TryReadResult_Error: return TryReadResult_Error;
    }

    switch (ParseGameDataTryReadKeyInt(gameData, state, "count", &value, 1, 1000))
    {
      case TryReadResult_Success:
        ASSURE_IS_NOT_INITIALIZED(count);
        wave->count = (uint16_t) value;
        continue;
      case TryReadResult_Error: return TryReadResult_Error;
    }

    switch (ParseGameDataTryReadKeyFloat(gameData, state, "delay", &wave->delay, 0.0f, 1000.0f))
    {
      case TryReadResult_Success: 
        ASSURE_IS_NOT_INITIALIZED(delay);
        continue;
      case TryReadResult_Error: return TryReadResult_Error;
    }

    switch (ParseGameDataTryReadKeyFloat(gameData, state, "interval", &wave->interval, 0.0f, 1000.0f))
    {
      case TryReadResult_Success: 
        ASSURE_IS_NOT_INITIALIZED(interval);
        continue;
      case TryReadResult_Error: return TryReadResult_Error;
    }

    int x, y;
    switch (ParseGameDataTryReadKeyIntVec2(gameData, state, "spawnPosition",
      &x, -10, 10, &y, -10, 10))
    {
      case TryReadResult_Success:
        ASSURE_IS_NOT_INITIALIZED(spawnPosition);
        wave->spawnPosition = (Vector2){x, y};
        continue;
      case TryReadResult_Error: return TryReadResult_Error;
    }

    switch (ParseGameDataTryReadEnemyTypeId(gameData, state, "enemyType", &wave->enemyType))
    {
      case TryReadResult_Success: 
        ASSURE_IS_NOT_INITIALIZED(enemyType);
        continue;
      case TryReadResult_Error: return TryReadResult_Error;
    }

    // no match, return to previous state and break
    *state = prevState;
    break;
  } 
#undef ASSURE_IS_NOT_INITIALIZED

#define ASSURE_IS_INITIALIZED(name) if (!name##Initialized) { \
  return ParseGameDataError(gameData, state, #name " not initialized"); }

  ASSURE_IS_INITIALIZED(wave);
  ASSURE_IS_INITIALIZED(count);
  ASSURE_IS_INITIALIZED(delay);
  ASSURE_IS_INITIALIZED(interval);
  ASSURE_IS_INITIALIZED(spawnPosition);
  ASSURE_IS_INITIALIZED(enemyType);

#undef ASSURE_IS_INITIALIZED

  return TryReadResult_Success;
}

TryReadResult ParseGameDataTryReadEnemyClassSection(ParsedGameData *gameData, ParserState *state)
{
  uint8_t enemyClassId;
  
  switch (ParseGameDataTryReadEnemyTypeId(gameData, state, "EnemyClass", &enemyClassId))
  {
    case TryReadResult_NoMatch: return TryReadResult_NoMatch;
    case TryReadResult_Error: return TryReadResult_Error;
  }

  EnemyClassConfig *enemyClass = &gameData->enemyClasses[enemyClassId];

#define FIELDS \
  HANDLE(speed, Float, 0.1f, 1000.0f) \
  HANDLE(health, Float, 1, 1000000) \
  HANDLE(radius, Float, 0.0f, 10.0f) \
  HANDLE(requiredContactTime, Float, 0.0f, 1000.0f) \
  HANDLE(maxAcceleration, Float, 0.0f, 1000.0f) \
  HANDLE(explosionDamage, Float, 0.0f, 1000.0f) \
  HANDLE(explosionRange, Float, 0.0f, 1000.0f) \
  HANDLE(explosionPushbackPower, Float, 0.0f, 1000.0f) \
  HANDLE(goldValue, Int, 1, 1000000)

#define HANDLE(name, type, min, max) int name##Initialized = 0;
  FIELDS
#undef HANDLE
  
  while (1)
  {
    ParserState prevState = *state;
    
    if (!ParserStateReadNextToken(state))
    {
      // end of file
      break;
    }

    #define HANDLE(name, type, min, max)\
      switch (ParseGameDataTryReadKey##type(gameData, state, #name, &enemyClass->name, min, max))\
      {\
        case TryReadResult_Success:\
          if (name##Initialized) {\
            return ParseGameDataError(gameData, state, #name " already initialized");\
          }\
          name##Initialized = 1;\
          continue;\
        case TryReadResult_Error: return TryReadResult_Error;\
      }
    
    FIELDS
    #undef HANDLE
    
    // no match, return to previous state and break
    *state = prevState;
    break;
  }

  #define HANDLE(name, type, min, max) \
    if (!name##Initialized) { \
      return ParseGameDataError(gameData, state, #name " not initialized"); \
    }
  
  FIELDS
  #undef HANDLE

  #undef FIELDS

  return TryReadResult_Success;
}

TryReadResult ParseGameDataTryReadLevelSection(ParsedGameData *gameData, ParserState *state)
{
  int levelId;
  TryReadResult result = ParseGameDataTryReadKeyInt(gameData, state, "Level", &levelId, 0, 31);
  if (result != TryReadResult_Success)
  {
    return result;
  }
  
  gameData->lastLevelIndex = levelId;
  Level *level = &gameData->levels[levelId];

  // since we require the initialGold to be initialized with at least 1, we can use it as a flag
  // to detect if the level was already initialized
  if (level->initialGold != 0)
  {
    return ParseGameDataError(gameData, state, TextFormat("level %d already initialized", levelId));
  }

  int initialGoldInitialized = 0;

  while (1)
  {
    // try to read the next token and if we don't know how to handle it,
    // we rewind and return
    ParserState prevState = *state;
    
    if (!ParserStateReadNextToken(state))
    {
      // end of file
      break;
    }

    switch (ParseGameDataTryReadKeyInt(gameData, state, "initialGold", &level->initialGold, 1, 100000))
    {
      case TryReadResult_Success: 
        if (initialGoldInitialized)
        {
          return ParseGameDataError(gameData, state, "initialGold already initialized");
        }
        initialGoldInitialized = 1;
        continue;
      case TryReadResult_Error: return TryReadResult_Error;
    }

    switch (ParseGameDataTryReadWaveSection(gameData, state))
    {
      case TryReadResult_Success: continue;
      case TryReadResult_Error: return TryReadResult_Error;
    }

    // no match, return to previous state and break
    *state = prevState;
    break;
  }

  if (!initialGoldInitialized)
  {
    return ParseGameDataError(gameData, state, "initialGold not initialized");
  }

  return TryReadResult_Success;
}

int ParseGameData(ParsedGameData *gameData, ParserState *state)
{
  *gameData = (ParsedGameData){0};
  gameData->lastLevelIndex = -1;

  while (ParserStateReadNextToken(state))
  {
    switch (ParseGameDataTryReadLevelSection(gameData, state))
    {
      case TryReadResult_Success: continue;
      case TryReadResult_Error: return 0;
    }

    switch (ParseGameDataTryReadEnemyClassSection(gameData, state))
    {
      case TryReadResult_Success: continue;
      case TryReadResult_Error: return 0;
    }
    // read other sections later


  }

  return 1;
}

#define EXPECT(cond, msg) if (!(cond)) { printf("Error %s:%d: %s\n", __FILE__, __LINE__, msg); failedCount++; } else { passedCount++; }

int RunParseTests()
{
  int passedCount = 0, failedCount = 0;
  ParserState state;
  ParsedGameData gameData = {0};

  state = (ParserState) {"Level 7\n  initialGold 100\nLevel 2 initialGold 200", 0, {0}};
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, &state) == 1, "Failed to parse level section");
  EXPECT(gameData.lastLevelIndex == 2, "lastLevelIndex should be 2");
  EXPECT(gameData.levels[7].initialGold == 100, "initialGold should be 100");
  EXPECT(gameData.levels[2].initialGold == 200, "initialGold should be 200");

  state = (ParserState) {"Level 392\n", 0, {0}};
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, &state) == 0, "Expected to fail parsing level section");
  EXPECT(TextFindIndex(gameData.parseError, "Invalid value range for Level") >= 0, "Expected invalid value range error");
  EXPECT(TextFindIndex(gameData.parseError, "line 1") >= 0, "Expected to find line number 1");
  
  state = (ParserState) {"Level 3\n initialGold -34", 0, {0}};
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, &state) == 0, "Expected to fail parsing level section");
  EXPECT(TextFindIndex(gameData.parseError, "Invalid value range for initialGold") >= 0, "Expected invalid value range error");
  EXPECT(TextFindIndex(gameData.parseError, "line 2") >= 0, "Expected to find line number 1");
  
  state = (ParserState) {"Level 3\n initialGold 2\n initialGold 3", 0, {0}};
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, &state) == 0, "Expected to fail parsing level section");
  EXPECT(TextFindIndex(gameData.parseError, "line 3") >= 0, "Expected to find line number 1");

  state = (ParserState) {"Level 3", 0, {0}};
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, &state) == 0, "Expected to fail parsing level section");

  state = (ParserState) {"Level 7\n  initialGold 100\nLevel 7 initialGold 200", 0, {0}};
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, &state) == 0, "Expected to fail parsing level section");
  EXPECT(TextFindIndex(gameData.parseError, "line 3") >= 0, "Expected to find line number 1");

  state = (ParserState) {
    "Level 7\n  initialGold 100\n"
    "Wave\n"
    "count 1 wave 2\n"
    "interval 0.5\n"
    "delay 1.0\n"
    "spawnPosition -3 4\n"
    "enemyType: ENEMY_TYPE_SHIELD"
    , 0, {0}};
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, &state) == 1, "Expected to succeed parsing level/wave section");
  EXPECT(gameData.levels[7].waves[0].count == 1, "Expected wave count to be 1");
  EXPECT(gameData.levels[7].waves[0].wave == 2, "Expected wave to be 2");
  EXPECT(gameData.levels[7].waves[0].interval == 0.5f, "Expected interval to be 0.5");
  EXPECT(gameData.levels[7].waves[0].delay == 1.0f, "Expected delay to be 1.0");
  EXPECT(gameData.levels[7].waves[0].spawnPosition.x == -3, "Expected spawnPosition.x to be 3");
  EXPECT(gameData.levels[7].waves[0].spawnPosition.y == 4, "Expected spawnPosition.y to be 4");
  EXPECT(gameData.levels[7].waves[0].enemyType == ENEMY_TYPE_SHIELD, "Expected enemyType to be ENEMY_TYPE_SHIELD");
  
  // for every entry in the wave section, we want to verify that if that value is 
  // missing, the parser will produce an error. We can do that by commenting out each
  // line individually in a loop - just replacing the two leading spaces with two dashes
  const char *testString =
    "Level 7 initialGold 100\n"
    "Wave\n"
    "  count 1\n"
    "  wave 2\n"
    "  interval 0.5\n"
    "  delay 1.0\n"
    "  spawnPosition 3 -4\n"
    "  enemyType: ENEMY_TYPE_SHIELD";
  for (int i = 0; testString[i]; i++)
  {
    if (testString[i] == '\n' && testString[i + 1] == ' ' && testString[i + 2] == ' ')
    {
      char copy[1024];
      strcpy(copy, testString);
      // commentify!
      copy[i + 1] = '-';
      copy[i + 2] = '-';
      state = (ParserState) {copy, 0, {0}};
      gameData = (ParsedGameData) {0};
      EXPECT(ParseGameData(&gameData, &state) == 0, "Expected to fail parsing level/wave section");
    }
  }

  // test wave section missing data / incorrect data

  state = (ParserState) {
    "Level 7\n  initialGold 100\n"
    "Wave\n"
    "count 1 wave 2\n"
    "interval 0.5\n"
    "delay 1.0\n"
    "spawnPosition -3\n" // missing y
    "enemyType: ENEMY_TYPE_SHIELD"
    , 0, {0}};
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, &state) == 0, "Expected to fail parsing level/wave section");
  EXPECT(TextFindIndex(gameData.parseError, "line 7") >= 0, "Expected to find line 7");
  
  state = (ParserState) {
    "Level 7\n  initialGold 100\n"
    "Wave\n"
    "count 1.0 wave 2\n"
    "interval 0.5\n"
    "delay 1.0\n"
    "spawnPosition -3\n" // missing y
    "enemyType: ENEMY_TYPE_SHIELD"
    , 0, {0}};
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, &state) == 0, "Expected to fail parsing level/wave section");
  EXPECT(TextFindIndex(gameData.parseError, "line 4") >= 0, "Expected to find line 3");

  // enemy class config parsing tests
  state = (ParserState) {
    "EnemyClass ENEMY_TYPE_MINION\n"
    "  health: 10.0\n"
    "  speed: 0.6\n"
    "  radius: 0.25\n"
    "  maxAcceleration: 1.0\n"
    "  explosionDamage: 1.0\n"
    "  requiredContactTime: 0.5\n"
    "  explosionRange: 1.0\n"
    "  explosionPushbackPower: 0.25\n"
    "  goldValue: 1\n"
    , 0, {0}};
  gameData = (ParsedGameData) {0};
  EXPECT(ParseGameData(&gameData, &state) == 1, "Expected to succeed parsing enemy class section");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].health == 10.0f, "Expected health to be 10.0");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].speed == 0.6f, "Expected speed to be 0.6");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].radius == 0.25f, "Expected radius to be 0.25");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].maxAcceleration == 1.0f, "Expected maxAcceleration to be 1.0");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].explosionDamage == 1.0f, "Expected explosionDamage to be 1.0");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].requiredContactTime == 0.5f, "Expected requiredContactTime to be 0.5");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].explosionRange == 1.0f, "Expected explosionRange to be 1.0");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].explosionPushbackPower == 0.25f, "Expected explosionPushbackPower to be 0.25");
  EXPECT(gameData.enemyClasses[ENEMY_TYPE_MINION].goldValue == 1, "Expected goldValue to be 1");

  testString =
    "EnemyClass ENEMY_TYPE_MINION\n"
    "  health: 10.0\n"
    "  speed: 0.6\n"
    "  radius: 0.25\n"
    "  maxAcceleration: 1.0\n"
    "  explosionDamage: 1.0\n"
    "  requiredContactTime: 0.5\n"
    "  explosionRange: 1.0\n"
    "  explosionPushbackPower: 0.25\n"
    "  goldValue: 1\n";
  for (int i = 0; testString[i]; i++)
  {
    if (testString[i] == '\n' && testString[i + 1] == ' ' && testString[i + 2] == ' ')
    {
      char copy[1024];
      strcpy(copy, testString);
      // commentify!
      copy[i + 1] = '-';
      copy[i + 2] = '-';
      state = (ParserState) {copy, 0, {0}};
      gameData = (ParsedGameData) {0};
      EXPECT(ParseGameData(&gameData, &state) == 0, "Expected to fail parsing EnemyClass section");
    }
  }


  printf("Passed %d test(s), Failed %d\n", passedCount, failedCount);

  return failedCount;
}

int main()
{
  printf("Running parse tests\n");
  if (RunParseTests())
  {
    return 1;
  }
  printf("\n");

  char *fileContent = LoadFileText("data/level.txt");
  if (fileContent == NULL)
  {
    printf("Failed to load file\n");
    return 1;
  }

  ParserState state = {fileContent, 0, {0}};
  ParsedGameData gameData = {0};

  if (!ParseGameData(&gameData, &state))
  {
    printf("Failed to parse game data: %s\n", gameData.parseError);
    UnloadFileText(fileContent);
    return 1;
  }

  UnloadFileText(fileContent);
  return 0; 
}
